from datetime import timedelta

from minio import Minio

from algo_flow.cores.config import settings


class MinioClient:
    def __init__(self):
        self.client = Minio(
            settings.minio.endpoint,
            access_key=settings.minio.access_key,
            secret_key=settings.minio.secret_key,
            secure=settings.minio.endpoint.startswith("https"),
        )
        self.bucket = settings.minio.bucket
        self._ensure_bucket()

    def _ensure_bucket(self):
        if not self.client.bucket_exists(self.bucket):
            self.client.make_bucket(self.bucket)

    def upload_file(self, file_path, object_name=None):
        if object_name is None:
            import os

            object_name = os.path.basename(file_path)
        self.client.fput_object(self.bucket, object_name, file_path)
        return object_name

    def download_file(self, object_name, file_path):
        self.client.fget_object(self.bucket, object_name, file_path)

    def get_presigned_url(self, object_name, expires=timedelta(seconds=3600)):
        return self.client.presigned_get_object(self.bucket, object_name, expires=expires)

    def remove_file(self, object_name):
        self.client.remove_object(self.bucket, object_name)


MINIO_CLIENT = MinioClient()
