import configparser
import os
import socket
from dataclasses import dataclass

from algo_flow.cores.log import LOG


def get_local_ip():
    """获取本机的局域网IP地址（非127.0.0.1）"""
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    try:
        # 这里的IP和端口不需要真实可达，只是触发系统自动选择本地IP
        s.connect(("8.8.8.8", 80))
        ip = s.getsockname()[0]
    except Exception as ex:
        LOG.error(ex)
        ip = "127.0.0.1"
    finally:
        s.close()
    LOG.debug(f"initialized local IP: {ip}")
    return ip


@dataclass
class AppConfig:
    project_name: str = "Algo Flow"
    api_version: str = "/api/v1"
    doc_path: str = "/docs"
    debug: bool = False
    logging_level: int = 20  # 0 NOTSET, 10 DEBUG, 20 INFO, 30 WARNING, 40 ERROR, 50 CRITICAL
    host: str = "0.0.0.0"
    port: int = 8000
    ip: str = None
    service_name: str = "algo-flow"
    environment: str = "dev"


@dataclass
class MySQLConfig:
    host: str
    port: int
    user: str
    password: str
    database: str

    @property
    def db_url(self):
        return f"mysql://{self.user}:{self.password}@{self.host}:{self.port}/{self.database}"

    @property
    def db_url_pymysql(self):
        return (
            f"mysql+pymysql://{self.user}:{self.password}@{self.host}:{self.port}/{self.database}"
        )


@dataclass
class RedisConfig:
    host: str
    port: int
    password: str
    default_db: int

    @property
    def db_url(self):
        return f"redis://:{self.password}@{self.host}:{self.port}/{self.default_db}"


@dataclass
class ZLMediaKitConfig:
    host: str = "127.0.0.1"
    secret: str = ""
    http_port: int = 8080
    rtsp_port: int = 8554
    rtmp_port: int = 1935


@dataclass
class SecurityConfig:
    secret_key: str
    algorithm: str
    token_expire_days: int


@dataclass
class GithubOAuthConfig:
    client: str
    secret: str


@dataclass
class FeishuConfig:
    webhook_url: str
    secret: str


@dataclass
class NacosConfig:
    server_addr: str
    namespace: str
    username: str
    password: str


@dataclass
class MinIOConfig:
    endpoint: str
    access_key: str
    secret_key: str
    bucket: str


@dataclass
class Settings:
    app: AppConfig
    mysql: MySQLConfig
    redis: RedisConfig
    zlmediakit: ZLMediaKitConfig
    security: SecurityConfig
    github: GithubOAuthConfig
    feishu: FeishuConfig
    nacos: NacosConfig
    minio: MinIOConfig


def get_config_path() -> str:
    """获取配置文件的路径，确保路径为绝对路径"""
    base_dir = os.path.dirname(os.path.abspath(__file__))
    config_file_path = os.getenv("CONFIG_FILE_PATH", "../../config.ini")
    # 如果配置文件路径不是绝对路径，则转换为绝对路径
    if not os.path.isabs(config_file_path):
        config_file_path = os.path.abspath(os.path.join(base_dir, config_file_path))
    # 如果配置文件不存在，则使用默认路径
    if not os.path.exists(config_file_path):
        config_file_path = os.path.abspath(os.path.join(base_dir, "../config.ini.example"))
    return config_file_path


def read_config() -> Settings:
    """读取配置文件并返回配置设置"""
    file_path = get_config_path()

    if not os.path.exists(file_path):
        raise FileNotFoundError(f"Configuration file not found: {file_path}")

    config = configparser.ConfigParser()
    config.read(file_path)

    app_config = AppConfig(**config["app"])
    app_config.debug = config.getboolean("app", "debug")
    app_config.logging_level = config.getint("app", "logging_level")
    app_config.port = config.getint("app", "port")
    # 读取网卡的IP地址
    if app_config.ip is None:
        app_config.ip = get_local_ip()

    mysql_config = MySQLConfig(**config["mysql"])
    redis_config = RedisConfig(**config["redis"])

    zlmediakit_config = ZLMediaKitConfig(**config["zlmediakit"])
    zlmediakit_config.http_port = config.getint("zlmediakit", "http_port")
    zlmediakit_config.rtmp_port = config.getint("zlmediakit", "rtmp_port")
    zlmediakit_config.rtsp_port = config.getint("zlmediakit", "rtsp_port")

    security_config = SecurityConfig(**config["security"])
    security_config.token_expire_days = config.getint("security", "token_expire_days")

    github_oauth_config = GithubOAuthConfig(**config["github"])
    feishu_config = FeishuConfig(**config["feishu"])

    nacos_config = NacosConfig(**config["nacos"])

    minio_config = MinIOConfig(**config["minio"])

    return Settings(
        app=app_config,
        mysql=mysql_config,
        redis=redis_config,
        zlmediakit=zlmediakit_config,
        security=security_config,
        github=github_oauth_config,
        feishu=feishu_config,
        nacos=nacos_config,
        minio=minio_config,
    )


settings = read_config()
