from typing import List

from fastapi import APIRouter, Security

from algo_flow.app.system.crud.user import get_user, get_user_menus, get_user_roles
from algo_flow.app.system.models import User
from algo_flow.app.system.schemas.menu import MenuResponse
from algo_flow.app.system.schemas.user import UserResponse, UserRoleResponse
from algo_flow.app.system.views.auth import get_current_active_user

user_me_router = APIRouter()


@user_me_router.get(
    "",
    response_model=UserResponse,
    summary="获取当前用户信息",
)
async def get_current_user_info(
    current_user: User = Security(get_current_active_user),
) -> UserResponse:
    """
    获取当前用户信息

    Returns:
        UserResponse: 当前用户的详细信息
    """
    user = await get_user(current_user.id)
    return UserResponse.model_validate(user)


@user_me_router.get(
    "/roles",
    response_model=UserRoleResponse,
    summary="获取当前用户角色",
)
async def get_current_user_roles(
    current_user: User = Security(get_current_active_user),
) -> UserRoleResponse:
    """
    获取当前用户角色

    Returns:
        UserRoleResponse: 当前用户的角色信息
    """
    user = await get_user_roles(current_user.id)
    return UserRoleResponse.model_validate(user)


@user_me_router.get(
    "/menus",
    response_model=List[MenuResponse],
    summary="获取当前用户菜单",
)
async def get_current_user_menus(
    current_user: User = Security(get_current_active_user),
) -> List[MenuResponse]:
    """
    获取当前用户菜单

    Returns:
        List[MenuResponse]: 当前用户的菜单列表
    """
    menus = await get_user_menus(current_user.id)
    return [MenuResponse.model_validate(menu) for menu in menus]
