from typing import Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security
from starlette import status

from algo_flow.app.system.crud.user import (
    assign_roles,
    count_users,
    create_user,
    delete_user,
    get_user,
    get_user_roles,
    list_users,
    update_password,
    update_user,
)
from algo_flow.app.system.schemas.user import (
    UserCreate,
    UserList,
    UserPasswordUpdate,
    UserResponse,
    UserRoleAssign,
    UserRoleResponse,
    UserUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

user_router = APIRouter()


@user_router.post(
    "",
    response_model=UserResponse,
    status_code=status.HTTP_201_CREATED,
    summary="创建用户",
    dependencies=[Security(get_current_active_user, scopes=["system:user:create"])],
)
async def create_user_api(user: UserCreate) -> UserResponse:
    """
    创建新用户

    Args:
        user: 用户创建参数

    Returns:
        UserResponse: 创建的用户信息

    Raises:
        HTTPException: 当用户名已存在时抛出 409 错误
    """
    try:
        db_user = await create_user(
            username=user.username,
            password=user.password,
            email=user.email,
            avatar=user.avatar,
            is_active=user.is_active,
            is_superuser=user.is_superuser,
            role_ids=user.role_ids,
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return UserResponse.model_validate(db_user)


@user_router.get(
    "/{user_id}",
    response_model=UserResponse,
    summary="获取用户详情",
    dependencies=[Security(get_current_active_user, scopes=["system:user:read"])],
)
async def get_user_api(
    user_id: UUID = Path(..., description="用户ID"),
) -> UserResponse:
    """
    获取用户详情

    Args:
        user_id: 用户ID

    Returns:
        UserResponse: 用户详细信息

    Raises:
        HTTPException: 当用户不存在时抛出 404 错误
    """
    try:
        user = await get_user(user_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return UserResponse.model_validate(user)


@user_router.get(
    "",
    response_model=UserList,
    summary="获取用户列表",
    dependencies=[Security(get_current_active_user, scopes=["system:user:read"])],
)
async def list_users_api(
    keyword: Optional[str] = Query(None, description="搜索关键字（用户名/昵称/邮箱/手机）"),
    is_active: Optional[bool] = Query(None, description="是否启用"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> UserList:
    """
    获取用户列表

    Args:
        keyword: 搜索关键字
        is_active: 是否启用
        offset: 分页偏移量，默认0
        limit: 分页大小，默认10，最大100

    Returns:
        UserList: 用户列表及总数
    """
    users = await list_users(
        keyword=keyword,
        is_active=is_active,
        offset=offset,
        limit=limit,
    )
    total = await count_users(keyword=keyword, is_active=is_active)

    return UserList(
        total=total,
        items=[UserResponse.model_validate(u) for u in users],
    )


@user_router.patch(
    "/{user_id}",
    response_model=UserResponse,
    summary="更新用户",
    dependencies=[Security(get_current_active_user, scopes=["system:user:update"])],
)
async def update_user_api(
    user_update: UserUpdate,
    user_id: UUID = Path(..., description="用户ID"),
) -> UserResponse:
    """
    更新用户信息

    Args:
        user_id: 用户ID
        user_update: 要更新的用户信息

    Returns:
        UserResponse: 更新后的用户信息

    Raises:
        HTTPException: 当用户不存在时抛出 404 错误，当用户名已存在时抛出 409 错误
    """
    try:
        user = await update_user(
            user_id=user_id,
            username=user_update.username,
            email=user_update.email,
            avatar=user_update.avatar,
            is_active=user_update.is_active,
            is_superuser=user_update.is_superuser,
            role_ids=user_update.role_ids,
        )
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return UserResponse.model_validate(user)


@user_router.delete(
    "/{user_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="删除用户",
    dependencies=[Security(get_current_active_user, scopes=["system:user:delete"])],
)
async def delete_user_api(
    user_id: UUID = Path(..., description="用户ID"),
) -> None:
    """
    删除用户

    Args:
        user_id: 用户ID

    Raises:
        HTTPException: 当用户不存在时抛出 404 错误
    """
    try:
        await delete_user(user_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))


@user_router.post(
    "/{user_id}/roles",
    response_model=UserRoleResponse,
    summary="分配用户角色",
    dependencies=[Security(get_current_active_user, scopes=["system:user:assign"])],
)
async def assign_user_roles_api(
    user_id: UUID = Path(..., description="用户ID"),
    roles: UserRoleAssign = None,
) -> UserRoleResponse:
    """
    分配用户角色

    Args:
        user_id: 用户ID
        roles: 角色ID列表

    Returns:
        UserRoleResponse: 用户角色信息

    Raises:
        HTTPException: 当用户不存在时抛出 404 错误
    """
    try:
        user = await assign_roles(user_id, roles.role_ids)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return UserRoleResponse.model_validate(user)


@user_router.get(
    "/{user_id}/roles",
    response_model=UserRoleResponse,
    summary="获取用户角色",
    dependencies=[Security(get_current_active_user, scopes=["system:user:read"])],
)
async def get_user_roles_api(
    user_id: UUID = Path(..., description="用户ID"),
) -> UserRoleResponse:
    """
    获取用户角色

    Args:
        user_id: 用户ID

    Returns:
        UserRoleResponse: 用户角色信息

    Raises:
        HTTPException: 当用户不存在时抛出 404 错误
    """
    try:
        user = await get_user_roles(user_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return UserRoleResponse.model_validate(user)


@user_router.patch(
    "/{user_id}/password",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="修改用户密码",
    dependencies=[Security(get_current_active_user, scopes=["system:user:update"])],
)
async def update_user_password_api(
    user_id: UUID = Path(..., description="用户ID"),
    password_update: UserPasswordUpdate = None,
) -> None:
    """
    修改用户密码

    Args:
        user_id: 用户ID
        password_update: 新密码信息

    Raises:
        HTTPException: 当用户不存在时抛出 404 错误
    """
    try:
        await update_password(user_id, password_update.new_password)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
