from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security
from starlette import status

from algo_flow.app.system.crud.role import (
    assign_permissions,
    count_roles,
    create_role,
    delete_role,
    get_role,
    get_role_permissions,
    list_roles,
    update_role,
)
from algo_flow.app.system.schemas.role import (
    RoleCreate,
    RoleList,
    RolePermissionAssign,
    RolePermissionResponse,
    RoleResponse,
    RoleUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

role_router = APIRouter()


@role_router.post(
    "",
    response_model=RoleResponse,
    status_code=status.HTTP_201_CREATED,
    summary="创建角色",
    dependencies=[Security(get_current_active_user, scopes=["system:role:create"])],
)
async def create_role_api(role: RoleCreate) -> RoleResponse:
    """
    创建新角色

    Args:
        role: 角色创建参数

    Returns:
        RoleResponse: 创建的角色信息

    Raises:
        HTTPException: 当角色标识已存在时抛出 409 错误
    """
    try:
        db_role = await create_role(
            name=role.name,
            identifier=role.identifier,
            description=role.description,
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return RoleResponse.model_validate(db_role)


@role_router.get(
    "/{role_id}",
    response_model=RoleResponse,
    summary="获取角色详情",
    dependencies=[Security(get_current_active_user, scopes=["system:role:read"])],
)
async def get_role_api(
    role_id: UUID = Path(..., description="角色ID"),
) -> RoleResponse:
    """
    获取角色详情

    Args:
        role_id: 角色ID

    Returns:
        RoleResponse: 角色详细信息

    Raises:
        HTTPException: 当角色不存在时抛出 404 错误
    """
    try:
        role = await get_role(role_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return RoleResponse.model_validate(role)


@role_router.get(
    "",
    response_model=RoleList,
    summary="获取角色列表",
    dependencies=[Security(get_current_active_user, scopes=["system:role:read"])],
)
async def list_roles_api(
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> RoleList:
    """
    获取角色列表

    Args:
        offset: 分页偏移量，默认0
        limit: 分页大小，默认10，最大100

    Returns:
        RoleList: 角色列表及总数
    """
    roles = await list_roles(offset=offset, limit=limit)
    total = await count_roles()

    return RoleList(
        total=total,
        items=[RoleResponse.model_validate(r) for r in roles],
    )


@role_router.patch(
    "/{role_id}",
    response_model=RoleResponse,
    summary="更新角色",
    dependencies=[Security(get_current_active_user, scopes=["system:role:update"])],
)
async def update_role_api(
    role_update: RoleUpdate,
    role_id: UUID = Path(..., description="角色ID"),
) -> RoleResponse:
    """
    更新角色信息

    Args:
        role_id: 角色ID
        role_update: 要更新的角色信息

    Returns:
        RoleResponse: 更新后的角色信息

    Raises:
        HTTPException: 当角色不存在时抛出 404 错误，当角色标识已存在时抛出 409 错误
    """
    try:
        role = await update_role(
            role_id=role_id,
            name=role_update.name,
            identifier=role_update.identifier,
            description=role_update.description,
        )
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return RoleResponse.model_validate(role)


@role_router.delete(
    "/{role_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="删除角色",
    dependencies=[Security(get_current_active_user, scopes=["system:role:delete"])],
)
async def delete_role_api(
    role_id: UUID = Path(..., description="角色ID"),
) -> None:
    """
    删除角色

    Args:
        role_id: 角色ID

    Raises:
        HTTPException: 当角色不存在时抛出 404 错误
    """
    try:
        await delete_role(role_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))


@role_router.post(
    "/{role_id}/permissions",
    response_model=RolePermissionResponse,
    summary="分配角色权限",
    dependencies=[Security(get_current_active_user, scopes=["system:role:assign"])],
)
async def assign_role_permissions_api(
    role_id: UUID = Path(..., description="角色ID"),
    permissions: RolePermissionAssign = ...,
) -> RolePermissionResponse:
    """
    分配角色权限

    Args:
        role_id: 角色ID
        permissions: 权限ID列表

    Returns:
        RolePermissionResponse: 角色权限信息

    Raises:
        HTTPException: 当角色不存在时抛出 404 错误
    """
    try:
        role = await assign_permissions(role_id, permissions.permission_ids)
        # 重新获取角色信息，确保包含最新的权限关系
        role = await get_role_permissions(role_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return RolePermissionResponse.model_validate(role)


@role_router.get(
    "/{role_id}/permissions",
    response_model=RolePermissionResponse,
    summary="获取角色权限",
    dependencies=[Security(get_current_active_user, scopes=["system:role:read"])],
)
async def get_role_permissions_api(
    role_id: UUID = Path(..., description="角色ID"),
) -> RolePermissionResponse:
    """
    获取角色权限

    Args:
        role_id: 角色ID

    Returns:
        RolePermissionResponse: 角色权限信息

    Raises:
        HTTPException: 当角色不存在时抛出 404 错误
    """
    try:
        role = await get_role_permissions(role_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return RolePermissionResponse.model_validate(role)
