from typing import Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security
from starlette import status

from algo_flow.app.system.crud.permission import (
    count_permissions,
    create_permission,
    delete_permission,
    get_permission,
    list_permissions,
    update_permission,
)
from algo_flow.app.system.schemas.permission import (
    PermissionCreate,
    PermissionList,
    PermissionResponse,
    PermissionUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

permission_router = APIRouter()


@permission_router.post(
    "",
    response_model=PermissionResponse,
    status_code=status.HTTP_201_CREATED,
    summary="创建权限",
    dependencies=[Security(get_current_active_user, scopes=["system:permission:create"])],
)
async def create_permission_api(permission: PermissionCreate) -> PermissionResponse:
    """
    创建新权限

    Args:
        permission: 权限创建参数

    Returns:
        PermissionResponse: 创建的权限信息

    Raises:
        HTTPException: 当权限标识已存在时抛出 409 错误
    """
    try:
        db_permission = await create_permission(
            name=permission.name,
            identifier=permission.identifier,
            module=permission.module,
            description=permission.description,
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return PermissionResponse.model_validate(db_permission)


@permission_router.get(
    "/{permission_id}",
    response_model=PermissionResponse,
    summary="获取权限详情",
    dependencies=[Security(get_current_active_user, scopes=["system:permission:read"])],
)
async def get_permission_api(
    permission_id: UUID = Path(..., description="权限ID"),
) -> PermissionResponse:
    """
    获取权限详情

    Args:
        permission_id: 权限ID

    Returns:
        PermissionResponse: 权限详细信息

    Raises:
        HTTPException: 当权限不存在时抛出 404 错误
    """
    try:
        permission = await get_permission(permission_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return PermissionResponse.model_validate(permission)


@permission_router.get(
    "",
    response_model=PermissionList,
    summary="获取权限列表",
    dependencies=[Security(get_current_active_user, scopes=["system:permission:read"])],
)
async def list_permissions_api(
    module: Optional[str] = Query(None, description="模块名称过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> PermissionList:
    """
    获取权限列表

    Args:
        module: 可选的模块名称过滤
        offset: 分页偏移量，默认0
        limit: 分页大小，默认10，最大100

    Returns:
        PermissionList: 权限列表及总数
    """
    permissions = await list_permissions(module=module, offset=offset, limit=limit)
    total = await count_permissions(module=module)

    return PermissionList(
        total=total,
        items=[PermissionResponse.model_validate(p) for p in permissions],
    )


@permission_router.patch(
    "/{permission_id}",
    response_model=PermissionResponse,
    summary="更新权限",
    dependencies=[Security(get_current_active_user, scopes=["system:permission:update"])],
)
async def update_permission_api(
    permission_update: PermissionUpdate,
    permission_id: UUID = Path(..., description="权限ID"),
) -> PermissionResponse:
    """
    更新权限信息

    Args:
        permission_id: 权限ID
        permission_update: 要更新的权限信息

    Returns:
        PermissionResponse: 更新后的权限信息

    Raises:
        HTTPException: 当权限不存在时抛出 404 错误，当权限标识已存在时抛出 409 错误
    """
    try:
        permission = await update_permission(
            permission_id=permission_id,
            name=permission_update.name,
            identifier=permission_update.identifier,
            module=permission_update.module,
            description=permission_update.description,
        )
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return PermissionResponse.model_validate(permission)


@permission_router.delete(
    "/{permission_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="删除权限",
    dependencies=[Security(get_current_active_user, scopes=["system:permission:delete"])],
)
async def delete_permission_api(
    permission_id: UUID = Path(..., description="权限ID"),
) -> None:
    """
    删除权限

    Args:
        permission_id: 权限ID

    Raises:
        HTTPException: 当权限不存在时抛出 404 错误
    """
    try:
        await delete_permission(permission_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
