from typing import List, Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security
from starlette import status

from algo_flow.app.system.crud.menu import (
    count_menus,
    create_menu,
    delete_menu,
    get_menu,
    get_menu_tree,
    list_menus,
    update_menu,
)
from algo_flow.app.system.schemas.menu import (
    MenuCreate,
    MenuList,
    MenuResponse,
    MenuUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

menu_router = APIRouter()


@menu_router.post(
    "",
    response_model=MenuResponse,
    status_code=status.HTTP_201_CREATED,
    summary="创建菜单",
    dependencies=[Security(get_current_active_user, scopes=["system:menu:create"])],
)
async def create_menu_api(menu: MenuCreate) -> MenuResponse:
    """
    创建新菜单

    Args:
        menu: 菜单创建参数

    Returns:
        MenuResponse: 创建的菜单信息

    Raises:
        HTTPException: 当菜单名称已存在时抛出 409 错误
    """
    try:
        db_menu = await create_menu(
            name=menu.name,
            path=menu.path,
            component=menu.component,
            locale=menu.locale,
            icon=menu.icon,
            requires_auth=menu.requires_auth,
            order=menu.order,
            hide_in_menu=menu.hide_in_menu,
            hide_children_in_menu=menu.hide_children_in_menu,
            no_affix=menu.no_affix,
            ignore_cache=menu.ignore_cache,
            parent_id=menu.parent_id,
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return MenuResponse.model_validate(db_menu)


@menu_router.get(
    "",
    response_model=MenuList,
    summary="获取菜单列表",
    dependencies=[Security(get_current_active_user, scopes=["system:menu:read"])],
)
async def list_menus_api(
    parent_id: Optional[UUID] = Query(None, description="父菜单ID过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> MenuList:
    """
    获取菜单列表

    Args:
        parent_id: 可选的父菜单ID过滤
        offset: 分页偏移量，默认0
        limit: 分页大小，默认10，最大100

    Returns:
        MenuList: 菜单列表及总数
    """
    menus = await list_menus(parent_id=parent_id, offset=offset, limit=limit)
    total = await count_menus(parent_id=parent_id)

    return MenuList(
        total=total,
        items=[MenuResponse.model_validate(m) for m in menus],
    )


@menu_router.get(
    "/tree",
    response_model=List[MenuResponse],
    summary="获取菜单树形结构",
    dependencies=[Security(get_current_active_user, scopes=["system:menu:read"])],
)
async def get_menu_tree_api() -> List[MenuResponse]:
    """
    获取菜单的树形结构

    Returns:
        List[MenuResponse]: 树形结构的菜单列表
    """
    menus = await get_menu_tree()
    return [MenuResponse.model_validate(menu) for menu in menus]


@menu_router.get(
    "/{menu_id}",
    response_model=MenuResponse,
    summary="获取菜单详情",
    dependencies=[Security(get_current_active_user, scopes=["system:menu:read"])],
)
async def get_menu_api(
    menu_id: UUID = Path(..., description="菜单ID"),
) -> MenuResponse:
    """
    获取菜单详情

    Args:
        menu_id: 菜单ID

    Returns:
        MenuResponse: 菜单详细信息

    Raises:
        HTTPException: 当菜单不存在时抛出 404 错误
    """
    try:
        menu = await get_menu(menu_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return MenuResponse.model_validate(menu)


@menu_router.patch(
    "/{menu_id}",
    response_model=MenuResponse,
    summary="更新菜单",
    dependencies=[Security(get_current_active_user, scopes=["system:menu:update"])],
)
async def update_menu_api(
    menu_update: MenuUpdate,
    menu_id: UUID = Path(..., description="菜单ID"),
) -> MenuResponse:
    """
    更新菜单信息

    Args:
        menu_id: 菜单ID
        menu_update: 要更新的菜单信息

    Returns:
        MenuResponse: 更新后的菜单信息

    Raises:
        HTTPException: 当菜单不存在时抛出 404 错误，当菜单名称已存在时抛出 409 错误
    """
    try:
        menu = await update_menu(
            menu_id=menu_id,
            name=menu_update.name,
            path=menu_update.path,
            component=menu_update.component,
            locale=menu_update.locale,
            icon=menu_update.icon,
            requires_auth=menu_update.requires_auth,
            order=menu_update.order,
            hide_in_menu=menu_update.hide_in_menu,
            hide_children_in_menu=menu_update.hide_children_in_menu,
            no_affix=menu_update.no_affix,
            ignore_cache=menu_update.ignore_cache,
            parent_id=menu_update.parent_id,
        )
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return MenuResponse.model_validate(menu)


@menu_router.delete(
    "/{menu_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="删除菜单",
    dependencies=[Security(get_current_active_user, scopes=["system:menu:delete"])],
)
async def delete_menu_api(
    menu_id: UUID = Path(..., description="菜单ID"),
) -> None:
    """
    删除菜单

    Args:
        menu_id: 菜单ID

    Raises:
        HTTPException: 当菜单不存在时抛出 404 错误
    """
    try:
        await delete_menu(menu_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
