from typing import Optional
from uuid import UUID

from fastapi import (
    APIRouter,
)
from fastapi import File as FastAPIFile
from fastapi import (
    HTTPException,
    Path,
    Query,
    Security,
    UploadFile,
    status,
)

from algo_flow.app.system.crud.file import (
    count_files,
    create_file,
    delete_file,
    get_file,
    list_files,
    update_file,
)
from algo_flow.app.system.schemas.file import (
    FileList,
    FileResponse,
    FileUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError
from algo_flow.cores.minio_client import MINIO_CLIENT

router = APIRouter()


@router.post(
    "",
    response_model=FileResponse,
    status_code=status.HTTP_201_CREATED,
    dependencies=[Security(get_current_active_user, scopes=["system:file:create"])],
    summary="上传文件",
)
async def create_file_api(
    file: UploadFile = FastAPIFile(...),
    description: Optional[str] = Query(None, description="文件描述"),
) -> FileResponse:
    import os
    import tempfile

    suffix = os.path.splitext(file.filename)[-1]
    with tempfile.NamedTemporaryFile(delete=False, suffix=suffix) as tmp:
        tmp.write(await file.read())
        tmp_path = tmp.name
    object_name = MINIO_CLIENT.upload_file(tmp_path)
    os.remove(tmp_path)
    try:
        db_file = await create_file(
            name=file.filename,
            path=object_name,
            size=file.size,
            type=file.content_type,
            description=description,
            status=1,
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))
    presigned_url = MINIO_CLIENT.get_presigned_url(object_name)
    resp = FileResponse.model_validate(db_file)
    resp = resp.model_copy(update={"presigned_url": presigned_url})
    return resp


@router.get(
    "/{file_id}",
    response_model=FileResponse,
    summary="获取文件详情",
    dependencies=[Security(get_current_active_user, scopes=["system:file:read"])],
)
async def get_file_api(file_id: UUID = Path(..., description="文件ID")) -> FileResponse:
    try:
        file_obj = await get_file(file_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    presigned_url = MINIO_CLIENT.get_presigned_url(file_obj.path)
    resp = FileResponse.model_validate(file_obj)
    resp = resp.model_copy(update={"presigned_url": presigned_url})
    return resp


@router.get(
    "",
    response_model=FileList,
    summary="获取文件列表",
    dependencies=[Security(get_current_active_user, scopes=["system:file:read"])],
)
async def list_files_api(
    status: Optional[int] = Query(None, description="文件状态过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> FileList:
    files = await list_files(status=status, offset=offset, limit=limit)
    total = await count_files(status=status)
    return FileList(total=total, items=[FileResponse.model_validate(f) for f in files])


@router.patch(
    "/{file_id}",
    response_model=FileResponse,
    summary="更新文件",
    dependencies=[Security(get_current_active_user, scopes=["system:file:update"])],
)
async def update_file_api(
    file_update: FileUpdate, file_id: UUID = Path(..., description="文件ID")
) -> FileResponse:
    update_fields = {}
    for field, value in file_update.model_dump(exclude_unset=True).items():
        if value is not None:
            update_fields[field] = value
    try:
        file_obj = await update_file(file_id=file_id, **update_fields)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))
    presigned_url = MINIO_CLIENT.get_presigned_url(file_obj.path)
    resp = FileResponse.model_validate(file_obj)
    resp = resp.model_copy(update={"presigned_url": presigned_url})
    return resp


@router.delete(
    "/{file_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="删除文件",
    dependencies=[Security(get_current_active_user, scopes=["system:file:delete"])],
)
async def delete_file_api(file_id: UUID = Path(..., description="文件ID")):
    try:
        await delete_file(file_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
