from typing import Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security
from starlette import status

from algo_flow.app.system.crud.config import (
    count_configs,
    create_config,
    delete_config,
    get_config,
    list_configs,
    update_config,
)
from algo_flow.app.system.schemas.config import (
    ConfigCreate,
    ConfigList,
    ConfigResponse,
    ConfigUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

config_router = APIRouter()


@config_router.post(
    "",
    response_model=ConfigResponse,
    status_code=status.HTTP_201_CREATED,
    summary="创建配置",
    dependencies=[Security(get_current_active_user, scopes=["system:config:create"])],
)
async def create_config_api(config: ConfigCreate) -> ConfigResponse:
    """
    创建新配置

    Args:
        config: 配置创建参数

    Returns:
        ConfigResponse: 创建的配置信息

    Raises:
        HTTPException: 当配置名称已存在时抛出 409 错误
    """
    try:
        db_config = await create_config(
            name=config.name,
            description=config.description,
            value=config.value,
            type=config.type,
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return ConfigResponse.model_validate(db_config)


@config_router.get(
    "/{config_id}",
    response_model=ConfigResponse,
    summary="获取配置详情",
    dependencies=[Security(get_current_active_user, scopes=["system:config:read"])],
)
async def get_config_api(
    config_id: UUID = Path(..., description="配置ID"),
) -> ConfigResponse:
    """
    获取配置详情

    Args:
        config_id: 配置ID

    Returns:
        ConfigResponse: 配置详细信息

    Raises:
        HTTPException: 当配置不存在时抛出 404 错误
    """
    try:
        config = await get_config(config_id=config_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return ConfigResponse.model_validate(config)


@config_router.get(
    "",
    response_model=ConfigList,
    summary="获取配置列表",
    dependencies=[Security(get_current_active_user, scopes=["system:config:read"])],
)
async def list_configs_api(
    type: Optional[str] = Query(None, description="配置类型过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> ConfigList:
    """
    获取配置列表

    Args:
        type: 可选的配置类型过滤
        offset: 分页偏移量，默认0
        limit: 分页大小，默认10，最大100

    Returns:
        ConfigList: 配置列表及总数
    """
    configs = await list_configs(type=type, offset=offset, limit=limit)
    total = await count_configs(type=type)

    return ConfigList(
        total=total,
        items=[ConfigResponse.model_validate(c) for c in configs],
    )


@config_router.patch(
    "/{config_id}",
    response_model=ConfigResponse,
    summary="更新配置",
    dependencies=[Security(get_current_active_user, scopes=["system:config:update"])],
)
async def update_config_api(
    config_update: ConfigUpdate,
    config_id: UUID = Path(..., description="配置ID"),
) -> ConfigResponse:
    """
    更新配置信息

    Args:
        config_id: 配置ID
        config_update: 要更新的配置信息

    Returns:
        ConfigResponse: 更新后的配置信息

    Raises:
        HTTPException: 当配置不存在时抛出 404 错误，当配置名称已存在时抛出 409 错误
    """
    try:
        config = await update_config(
            config_id=config_id,
            name=config_update.name,
            description=config_update.description,
            value=config_update.value,
            type=config_update.type,
        )
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return ConfigResponse.model_validate(config)


@config_router.delete(
    "/{config_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="删除配置",
    dependencies=[Security(get_current_active_user, scopes=["system:config:delete"])],
)
async def delete_config_api(
    config_id: UUID = Path(..., description="配置ID"),
) -> None:
    """
    删除配置

    Args:
        config_id: 配置ID

    Raises:
        HTTPException: 当配置不存在时抛出 404 错误
    """
    try:
        await delete_config(config_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
