from datetime import datetime
from typing import Dict, List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, EmailStr, Field, field_validator

from algo_flow.app.system.schemas.role import RoleResponse


class UserBase(BaseModel):
    """用户基础模型"""

    username: str = Field(..., description="用户名", min_length=1, max_length=50)
    email: Optional[EmailStr] = Field(None, description="邮箱", min_length=1, max_length=50)
    avatar: Optional[str] = Field(None, description="头像")
    is_active: bool = Field(default=True, description="是否激活")
    is_superuser: bool = Field(default=False, description="是否超级管理员")


class UserCreate(UserBase):
    """用户创建请求模型"""

    password: str = Field(..., description="密码", min_length=6, max_length=32)
    role_ids: Optional[List[UUID]] = Field(default_factory=list, description="角色ID列表")


class UserUpdate(BaseModel):
    """用户更新请求模型"""

    username: Optional[str] = Field(None, description="用户名", min_length=1, max_length=50)
    email: Optional[EmailStr] = Field(None, description="邮箱", min_length=1, max_length=50)
    avatar: Optional[str] = Field(None, description="头像")
    is_active: Optional[bool] = Field(None, description="是否激活")
    is_superuser: Optional[bool] = Field(None, description="是否超级管理员")
    role_ids: Optional[List[UUID]] = Field(None, description="角色ID列表")


class UserResponse(UserBase):
    """用户响应模型"""

    id: UUID = Field(..., description="用户ID")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")

    model_config = ConfigDict(from_attributes=True)


class UserList(BaseModel):
    """用户列表响应模型"""

    total: int = Field(..., description="总数")
    items: List[UserResponse] = Field(..., description="用户列表")


class UserRoleAssign(BaseModel):
    """用户角色分配模型"""

    role_ids: List[UUID] = Field(..., description="角色ID列表")


class UserRoleResponse(UserResponse):
    """用户角色响应模型"""

    roles: List[RoleResponse] = Field(..., description="角色列表")


class UserPasswordUpdate(BaseModel):
    """用户密码更新模型"""

    new_password: str = Field(..., description="新密码", min_length=6)
    confirm_password: str = Field(..., description="确认新密码", min_length=6)

    @field_validator("confirm_password")
    def passwords_match(cls, v: str, values: Dict) -> str:
        if "new_password" in values and v != values["new_password"]:
            raise ValueError("两次输入的新密码不匹配")
        return v
