from datetime import datetime
from typing import List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field

from algo_flow.app.system.schemas.menu import MenuResponse
from algo_flow.app.system.schemas.permission import PermissionResponse


class RoleBase(BaseModel):
    """角色基础模型"""

    name: str = Field(..., description="角色名称", min_length=1, max_length=50)
    identifier: str = Field(..., description="角色标识符", min_length=1, max_length=50)
    description: Optional[str] = Field(None, description="角色描述")
    is_active: bool = Field(default=True, description="是否启用")
    is_default: bool = Field(default=False, description="是否为默认角色")


class RoleCreate(RoleBase):
    """创建角色模型"""

    permission_ids: Optional[List[UUID]] = Field(default_factory=list, description="权限ID列表")
    menu_ids: Optional[List[UUID]] = Field(default_factory=list, description="菜单ID列表")


class RoleUpdate(BaseModel):
    """更新角色模型"""

    name: Optional[str] = Field(None, description="角色名称", min_length=1, max_length=50)
    identifier: Optional[str] = Field(None, description="角色标识符", min_length=1, max_length=50)
    description: Optional[str] = Field(None, description="角色描述")
    is_active: Optional[bool] = Field(None, description="是否启用")
    is_default: Optional[bool] = Field(None, description="是否为默认角色")
    permission_ids: Optional[List[UUID]] = Field(None, description="权限ID列表")
    menu_ids: Optional[List[UUID]] = Field(None, description="菜单ID列表")


class RoleResponse(RoleBase):
    """角色响应模型"""

    id: UUID = Field(..., description="角色ID")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")
    permissions: List[PermissionResponse] = Field(default=[], description="角色权限")
    menus: List[MenuResponse] = Field(default=[], description="角色菜单")

    model_config = ConfigDict(from_attributes=True)

    @classmethod
    def model_validate(cls, obj, *, context=None, **kwargs):
        """自定义验证方法，处理 ManyToMany 关系字段"""
        if hasattr(obj, "__dict__"):
            obj_dict = obj.__dict__.copy()
            # 处理 permissions 字段
            if hasattr(obj, "permissions"):
                obj_dict["permissions"] = [p for p in obj.permissions]
            # 处理 menus 字段
            if hasattr(obj, "menus"):
                obj_dict["menus"] = [m for m in obj.menus]
            return super().model_validate(obj_dict, context=context, **kwargs)
        return super().model_validate(obj, context=context, **kwargs)


class RoleList(BaseModel):
    """角色列表响应模型"""

    total: int = Field(..., description="总数")
    items: List[RoleResponse] = Field(..., description="角色列表")


class PermissionInfo(BaseModel):
    """权限信息模型"""

    model_config = ConfigDict(from_attributes=True)

    id: UUID
    name: str
    identifier: str
    module: str
    description: Optional[str] = None


class RolePermissionResponse(RoleResponse):
    """角色权限响应模型"""

    permissions: List[PermissionInfo] = Field(..., description="权限列表")

    @classmethod
    def model_validate(cls, obj, *, context=None, **kwargs):
        """自定义验证方法，处理 ManyToMany 关系字段"""
        if hasattr(obj, "__dict__"):
            obj_dict = obj.__dict__.copy()
            # 处理 permissions 字段
            if hasattr(obj, "permissions"):
                obj_dict["permissions"] = [p for p in obj.permissions]
            # 处理 menus 字段
            if hasattr(obj, "menus"):
                obj_dict["menus"] = []  # 权限响应不需要菜单信息
            return super().model_validate(obj_dict, context=context, **kwargs)
        return super().model_validate(obj, context=context, **kwargs)


class RolePermissionAssign(BaseModel):
    """角色权限分配模型"""

    permission_ids: List[UUID]
