from datetime import datetime
from typing import List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field


class PermissionBase(BaseModel):
    """权限基础模型"""

    name: str = Field(..., description="权限名称", min_length=1, max_length=100)
    identifier: str = Field(..., description="权限标识符", min_length=1, max_length=100)
    module: str = Field(..., description="所属模块", pattern="^(api|page|button)$")
    description: Optional[str] = Field(None, description="权限描述")
    is_active: bool = Field(default=True, description="是否启用")


class PermissionCreate(PermissionBase):
    """权限创建请求模型"""


class PermissionUpdate(BaseModel):
    """权限更新请求模型"""

    name: Optional[str] = Field(None, description="权限名称", min_length=1, max_length=100)
    identifier: Optional[str] = Field(None, description="权限标识符", min_length=1, max_length=100)
    module: Optional[str] = Field(None, description="所属模块", pattern="^(api|page|button)$")
    description: Optional[str] = Field(None, description="权限描述")
    is_active: Optional[bool] = Field(None, description="是否启用")


class PermissionResponse(PermissionBase):
    """权限响应模型"""

    id: UUID = Field(..., description="权限ID")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")

    model_config = ConfigDict(from_attributes=True)


class PermissionList(BaseModel):
    """权限列表响应模型"""

    total: int = Field(..., description="总数")
    items: List[PermissionResponse] = Field(..., description="权限列表")
