from datetime import datetime
from typing import List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field


class FileBase(BaseModel):
    name: str = Field(..., description="文件名", min_length=1, max_length=255)
    description: Optional[str] = Field(None, description="文件描述")
    status: int = Field(1, description="文件状态：1-正常，0-删除")
    path: str = Field(..., description="MinIO 对象存储路径")
    size: int = Field(..., description="文件大小（字节）")
    type: str = Field(..., description="文件类型（MIME）")


class FileCreate(FileBase):
    pass


class FileUpdate(BaseModel):
    name: Optional[str] = Field(None, description="文件名", min_length=1, max_length=255)
    description: Optional[str] = Field(None, description="文件描述")
    status: Optional[int] = Field(None, description="文件状态")
    path: Optional[str] = Field(None, description="MinIO 对象存储路径")
    size: Optional[int] = Field(None, description="文件大小（字节）")
    type: Optional[str] = Field(None, description="文件类型（MIME）")


class FileResponse(FileBase):
    id: UUID = Field(..., description="文件ID")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")
    presigned_url: Optional[str] = Field(None, description="文件下载链接")
    model_config = ConfigDict(from_attributes=True)


class FileList(BaseModel):
    total: int = Field(..., description="总数")
    items: List[FileResponse] = Field(..., description="文件列表")
