from tortoise import fields

from algo_flow.cores.model import Model


class User(Model):
    """用户模型"""

    username = fields.CharField(max_length=50, unique=True, description="用户名")
    email = fields.CharField(max_length=50, unique=True, description="邮箱")
    hashed_password = fields.CharField(max_length=255, description="密码哈希")
    avatar = fields.CharField(max_length=255, null=True, description="头像")
    is_active = fields.BooleanField(default=True, description="是否激活")
    is_superuser = fields.BooleanField(default=False, description="是否超级管理员")
    roles = fields.ManyToManyField(
        "models.Role",
        related_name="users",
        through="system_user_roles",
        description="用户角色",
    )

    class Meta:
        table = "system_users"
        table_description = "用户表"


class Role(Model):
    """角色模型"""

    name = fields.CharField(max_length=50, unique=True, description="角色名称")
    identifier = fields.CharField(max_length=50, unique=True, description="角色标识符")
    description = fields.TextField(null=True, description="角色描述")
    is_active = fields.BooleanField(default=True, description="是否启用")
    is_default = fields.BooleanField(default=False, description="是否为默认角色")
    permissions = fields.ManyToManyField(
        "models.Permission",
        related_name="roles",
        through="system_role_permissions",
        description="角色权限",
    )
    menus = fields.ManyToManyField(
        "models.Menu",
        related_name="roles",
        through="system_role_menus",
        description="角色菜单",
    )

    class Meta:
        table = "system_roles"
        table_description = "角色表"


class Permission(Model):
    """权限模型"""

    name = fields.CharField(max_length=100, unique=True, description="权限名称")
    identifier = fields.CharField(max_length=100, unique=True, description="权限标识符")
    module = fields.CharField(max_length=50, description="所属模块")
    description = fields.TextField(null=True, description="权限描述")

    class Meta:
        table = "system_permissions"
        table_description = "权限表"


class Menu(Model):
    """菜单模型"""

    name = fields.CharField(max_length=50, unique=True, description="菜单名称")
    path = fields.CharField(max_length=50, unique=True, description="菜单路径")
    component = fields.CharField(max_length=50, description="组件路径")
    locale = fields.CharField(max_length=50, null=True, description="国际化")
    redirect = fields.CharField(max_length=50, null=True, description="重定向路径")
    icon = fields.CharField(max_length=50, null=True, description="图标")
    requires_auth = fields.BooleanField(default=False, description="是否需要权限")
    order = fields.IntField(default=0, description="排序")
    hide_in_menu = fields.BooleanField(default=False, description="是否隐藏在菜单中")
    hide_children_in_menu = fields.BooleanField(default=False, description="是否隐藏子菜单")
    no_affix = fields.BooleanField(default=False, description="是否不固定在标签栏")
    ignore_cache = fields.BooleanField(default=False, description="是否不缓存")
    parent_id = fields.UUIDField(null=True, description="父菜单ID")

    class Meta:
        table = "system_menus"
        table_description = "菜单表"


class File(Model):
    """文件模型

    用于管理存储在 MinIO 上的文件元数据。
    """

    id = fields.UUIDField(primary_key=True, description="文件唯一标识符")
    name = fields.CharField(max_length=255, description="文件名")
    path = fields.CharField(max_length=255, description="MinIO 对象存储路径")
    size = fields.BigIntField(description="文件大小（字节）")
    type = fields.CharField(max_length=50, description="文件类型（MIME）")
    status = fields.IntField(default=1, description="文件状态：1-正常，0-删除")
    description = fields.TextField(null=True, description="文件描述")
    created_at = fields.DatetimeField(auto_now_add=True, description="上传时间")
    updated_at = fields.DatetimeField(auto_now=True, description="更新时间")

    class Meta:
        table = "system_files"
        ordering = ["-updated_at"]


class Config(Model):
    """配置模型"""

    name = fields.CharField(max_length=100, unique=True, description="配置名称")
    description = fields.TextField(null=True, description="配置描述")
    value = fields.JSONField(description="配置值")
    type = fields.CharField(max_length=50, description="配置类型")

    class Meta:
        table = "system_configs"
        table_description = "系统配置表"
