from typing import List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.expressions import Q
from tortoise.transactions import atomic

from algo_flow.app.system.models import Menu, Role, User
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError
from algo_flow.cores.pwd import get_password_hash


async def create_user(
    username: str,
    password: str,
    email: Optional[str] = None,
    avatar: Optional[str] = None,
    is_active: bool = True,
    is_superuser: bool = False,
    role_ids: Optional[List[UUID]] = None,
) -> User:
    """
    创建新用户

    Args:
        username: 用户名
        password: 密码
        email: 邮箱
        avatar: 头像
        is_active: 是否启用
        is_superuser: 是否为超级管理员
        role_ids: 角色ID列表

    Returns:
        User: 创建的用户实例

    Raises:
        ResourceConflictError: 当用户名已存在时抛出
    """
    # 检查用户名是否已存在
    existing = await User.filter(username=username).first()
    if existing:
        raise ResourceConflictError(f"用户名 '{username}' 已存在")

    # 检查邮箱是否已存在
    if email:
        existing = await User.filter(email=email).first()
        if existing:
            raise ResourceConflictError(f"邮箱 '{email}' 已被使用")

    # 创建新用户
    user = await User.create(
        username=username,
        hashed_password=get_password_hash(password),
        email=email,
        avatar=avatar,
        is_active=is_active,
        is_superuser=is_superuser,
    )

    # 分配角色
    if role_ids:
        roles = []
        for role_id in role_ids:
            try:
                role = await Role.get(id=role_id)
                roles.append(role)
            except DoesNotExist:
                raise ResourceNotFoundError(f"角色 {role_id} 不存在")
        await user.roles.add(*roles)

    # 重新获取包含角色的用户实例
    return await User.get(id=user.id).prefetch_related("roles")


async def get_user(user_id: UUID) -> User:
    """
    获取用户详情

    Args:
        user_id: 用户ID

    Returns:
        User: 用户实例

    Raises:
        ResourceNotFoundError: 当用户不存在时抛出
    """
    try:
        user = await User.get(id=user_id).prefetch_related("roles")
    except DoesNotExist:
        raise ResourceNotFoundError(f"用户 {user_id} 不存在")

    return user


async def list_users(
    keyword: Optional[str] = None,
    is_active: Optional[bool] = None,
    offset: int = 0,
    limit: int = 10,
) -> List[User]:
    """
    获取用户列表

    Args:
        keyword: 搜索关键字（用户名/昵称/邮箱/手机）
        is_active: 是否启用
        offset: 分页偏移量
        limit: 分页大小

    Returns:
        List[User]: 用户列表
    """
    query = User.all()

    if keyword:
        query = query.filter(Q(username__icontains=keyword) | Q(email__icontains=keyword))

    if is_active is not None:
        query = query.filter(is_active=is_active)

    # 按用户名排序
    query = query.order_by("username")

    users = await query.offset(offset).limit(limit)
    return users


async def _validate_username(user: User, new_username: str) -> None:
    """验证用户名是否可用"""
    if new_username != user.username:
        existing = await User.filter(username=new_username).first()
        if existing:
            raise ResourceConflictError(f"用户名 '{new_username}' 已存在")


async def _validate_email(user: User, new_email: str) -> None:
    """验证邮箱是否可用"""
    if new_email != user.email:
        existing = await User.filter(email=new_email).first()
        if existing:
            raise ResourceConflictError(f"邮箱 '{new_email}' 已被使用")


async def _update_user_unique_fields(
    user: User,
    username: Optional[str],
    email: Optional[str],
) -> None:
    """更新用户唯一字段"""
    if username is not None:
        await _validate_username(user, username)
        user.username = username
    if email is not None:
        await _validate_email(user, email)
        user.email = email


async def _update_user_basic_fields(
    user: User,
    avatar: Optional[str],
    is_active: Optional[bool],
    is_superuser: Optional[bool],
) -> None:
    """更新用户基本字段"""
    if avatar is not None:
        user.avatar = avatar
    if is_active is not None:
        user.is_active = is_active
    if is_superuser is not None:
        user.is_superuser = is_superuser


async def _update_user_roles(
    user: User,
    role_ids: List[UUID],
) -> None:
    """更新用户角色"""
    roles = []
    for role_id in role_ids:
        try:
            role = await Role.get(id=role_id)
            roles.append(role)
        except DoesNotExist:
            raise ResourceNotFoundError(f"角色 {role_id} 不存在")
    await user.roles.clear()
    await user.roles.add(*roles)


async def update_user(
    user_id: UUID,
    username: Optional[str] = None,
    email: Optional[str] = None,
    avatar: Optional[str] = None,
    is_active: Optional[bool] = None,
    is_superuser: Optional[bool] = None,
    role_ids: Optional[List[UUID]] = None,
) -> User:
    """
    更新用户信息

    Args:
        user_id: 用户ID
        username: 新的用户名
        email: 新的邮箱
        avatar: 新的头像
        is_active: 是否启用
        is_superuser: 是否为超级管理员
        role_ids: 新的角色ID列表

    Returns:
        User: 更新后的用户实例

    Raises:
        ResourceNotFoundError: 当用户不存在时抛出
        ResourceConflictError: 当用户名/邮箱已存在时抛出
    """
    try:
        user = await User.get(id=user_id).prefetch_related("roles")
    except DoesNotExist:
        raise ResourceNotFoundError(f"用户 {user_id} 不存在")

    await _update_user_unique_fields(user, username, email)
    await _update_user_basic_fields(user, avatar, is_active, is_superuser)

    if role_ids is not None:
        await _update_user_roles(user, role_ids)

    await user.save()
    return user


@atomic()
async def delete_user(user_id: UUID) -> None:
    """
    删除用户

    Args:
        user_id: 用户ID

    Raises:
        ResourceNotFoundError: 当用户不存在时抛出
    """
    try:
        user = await User.get(id=user_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"用户 {user_id} 不存在")

    # 删除用户（这里会自动处理用户角色关系）
    await user.delete()


async def count_users(
    keyword: Optional[str] = None,
    is_active: Optional[bool] = None,
) -> int:
    """
    获取用户总数

    Args:
        keyword: 搜索关键字（用户名/昵称/邮箱/手机）
        is_active: 是否启用

    Returns:
        int: 用户总数
    """
    query = User.all()

    if keyword:
        query = query.filter(Q(username__icontains=keyword) | Q(email__icontains=keyword))

    if is_active is not None:
        query = query.filter(is_active=is_active)

    return await query.count()


@atomic()
async def assign_roles(user_id: UUID, role_ids: List[UUID]) -> User:
    """
    分配用户角色

    Args:
        user_id: 用户ID
        role_ids: 角色ID列表

    Returns:
        User: 更新后的用户实例，包含角色关系

    Raises:
        ResourceNotFoundError: 当用户或角色不存在时抛出
    """
    try:
        user = await User.get(id=user_id).prefetch_related("roles")
    except DoesNotExist:
        raise ResourceNotFoundError(f"用户 {user_id} 不存在")

    # 验证所有角色是否存在
    roles = []
    for role_id in role_ids:
        try:
            role = await Role.get(id=role_id)
            roles.append(role)
        except DoesNotExist:
            raise ResourceNotFoundError(f"角色 {role_id} 不存在")

    # 清除现有角色并设置新角色
    await user.roles.clear()
    await user.roles.add(*roles)

    # 重新获取包含角色的用户实例
    return await User.get(id=user_id).prefetch_related("roles")


async def get_user_roles(user_id: UUID) -> User:
    """
    获取用户角色

    Args:
        user_id: 用户ID

    Returns:
        User: 用户实例，包含角色关系

    Raises:
        ResourceNotFoundError: 当用户不存在时抛出
    """
    try:
        user = await User.get(id=user_id).prefetch_related("roles")
    except DoesNotExist:
        raise ResourceNotFoundError(f"用户 {user_id} 不存在")

    return user


async def update_password(user_id: UUID, password: str) -> None:
    """
    更新用户密码

    Args:
        user_id: 用户ID
        password: 新密码

    Raises:
        ResourceNotFoundError: 当用户不存在时抛出
    """
    try:
        user = await User.get(id=user_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"用户 {user_id} 不存在")

    user.password = get_password_hash(password)
    await user.save()


async def get_user_menus(user_id: UUID) -> List[Menu]:
    """
    获取用户菜单

    Args:
        user_id: 用户ID

    Returns:
        List[Menu]: 用户的所有菜单列表，按照排序顺序排列

    Raises:
        ResourceNotFoundError: 当用户不存在时抛出
    """
    # 超级管理员直接返回所有菜单
    try:
        user = await User.get(id=user_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"用户 {user_id} 不存在")
    if user.is_superuser:
        return await Menu.all().order_by("order", "name")

    try:
        user = await User.get(id=user_id).prefetch_related("roles__menus")
    except DoesNotExist:
        raise ResourceNotFoundError(f"用户 {user_id} 不存在")

    # 收集所有菜单
    menu_set = set()
    for role in user.roles:
        for menu in role.menus:
            menu_set.add(menu.id)

    # 获取排序后的菜单列表
    menus = await Menu.filter(id__in=menu_set).order_by("order", "name")
    return menus
