from typing import List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.system.models import Permission, Role
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_role(
    name: str,
    identifier: str,
    description: Optional[str] = None,
) -> Role:
    """
    创建新角色

    Args:
        name: 角色名称
        identifier: 角色标识
        description: 角色描述

    Returns:
        Role: 创建的角色实例

    Raises:
        ResourceConflictError: 当角色标识已存在时抛出
    """
    # 检查角色标识是否已存在
    existing = await Role.filter(identifier=identifier).first()
    if existing:
        raise ResourceConflictError(f"角色标识 '{identifier}' 已存在")

    # 创建新角色
    role = await Role.create(
        name=name,
        identifier=identifier,
        description=description,
    )

    # 预加载关联字段
    await role.fetch_related("permissions", "menus")
    return role


async def get_role(role_id: UUID) -> Role:
    """
    获取角色详情

    Args:
        role_id: 角色ID

    Returns:
        Role: 角色实例

    Raises:
        ResourceNotFoundError: 当角色不存在时抛出
    """
    try:
        role = await Role.get(id=role_id).prefetch_related("permissions", "menus")
    except DoesNotExist:
        raise ResourceNotFoundError(f"角色 {role_id} 不存在")

    return role


async def list_roles(
    offset: int = 0,
    limit: int = 10,
) -> List[Role]:
    """
    获取角色列表

    Args:
        offset: 分页偏移量
        limit: 分页大小

    Returns:
        List[Role]: 角色列表
    """
    query = Role.all()

    # 按标识符排序
    query = query.order_by("identifier")

    # 预加载关联字段
    query = query.prefetch_related("permissions", "menus")

    roles = await query.offset(offset).limit(limit)
    return roles


async def _validate_role_identifier(role: Role, new_identifier: str) -> None:
    """验证角色标识是否可用"""
    if new_identifier != role.identifier:
        existing = await Role.filter(identifier=new_identifier).first()
        if existing:
            raise ResourceConflictError(f"角色标识 '{new_identifier}' 已存在")


async def update_role(
    role_id: UUID,
    name: Optional[str] = None,
    identifier: Optional[str] = None,
    description: Optional[str] = None,
) -> Role:
    """
    更新角色信息

    Args:
        role_id: 角色ID
        name: 新的角色名称
        identifier: 新的角色标识
        description: 新的角色描述

    Returns:
        Role: 更新后的角色实例

    Raises:
        ResourceNotFoundError: 当角色不存在时抛出
        ResourceConflictError: 当新角色标识已存在时抛出
    """
    try:
        role = await Role.get(id=role_id).prefetch_related("permissions", "menus")
    except DoesNotExist:
        raise ResourceNotFoundError(f"角色 {role_id} 不存在")

    # 验证角色标识
    if identifier is not None:
        await _validate_role_identifier(role, identifier)
        role.identifier = identifier

    # 更新其他字段
    if name is not None:
        role.name = name
    if description is not None:
        role.description = description

    await role.save()
    await role.fetch_related("permissions", "menus")
    return role


@atomic()
async def delete_role(role_id: UUID) -> None:
    """
    删除角色

    Args:
        role_id: 角色ID

    Raises:
        ResourceNotFoundError: 当角色不存在时抛出
    """
    try:
        role = await Role.get(id=role_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"角色 {role_id} 不存在")

    # 删除角色（这里会自动处理角色权限关系和用户角色关系）
    await role.delete()


async def count_roles() -> int:
    """
    获取角色总数

    Returns:
        int: 角色总数
    """
    return await Role.all().count()


@atomic()
async def assign_permissions(role_id: UUID, permission_ids: List[UUID]) -> Role:
    """
    分配角色权限

    Args:
        role_id: 角色ID
        permission_ids: 权限ID列表

    Returns:
        Role: 更新后的角色实例，包含权限关系

    Raises:
        ResourceNotFoundError: 当角色或权限不存在时抛出
    """
    try:
        role = await Role.get(id=role_id).prefetch_related("permissions")
    except DoesNotExist:
        raise ResourceNotFoundError(f"角色 {role_id} 不存在")

    # 验证所有权限是否存在
    permissions = []
    for permission_id in permission_ids:
        try:
            permission = await Permission.get(id=permission_id)
            permissions.append(permission)
        except DoesNotExist:
            raise ResourceNotFoundError(f"权限 {permission_id} 不存在")

    # 清除现有权限并设置新权限
    await role.permissions.clear()
    await role.permissions.add(*permissions)

    # 重新获取角色，确保包含最新的权限关系
    return await Role.get(id=role_id).prefetch_related("permissions")


async def get_role_permissions(role_id: UUID) -> Role:
    """
    获取角色权限

    Args:
        role_id: 角色ID

    Returns:
        Role: 角色实例，包含权限关系

    Raises:
        ResourceNotFoundError: 当角色不存在时抛出
    """
    try:
        role = await Role.get(id=role_id).prefetch_related("permissions")
    except DoesNotExist:
        raise ResourceNotFoundError(f"角色 {role_id} 不存在")

    return role
