from typing import List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.system.models import Permission
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_permission(
    name: str,
    identifier: str,
    module: str,
    description: Optional[str] = None,
) -> Permission:
    """
    创建新权限

    Args:
        name: 权限名称
        identifier: 权限标识
        module: 所属模块
        description: 权限描述

    Returns:
        Permission: 创建的权限实例

    Raises:
        ResourceConflictError: 当权限标识已存在时抛出
    """
    # 检查权限标识是否已存在
    existing = await Permission.filter(identifier=identifier).first()
    if existing:
        raise ResourceConflictError(f"权限标识 '{identifier}' 已存在")

    # 创建新权限
    permission = await Permission.create(
        name=name,
        identifier=identifier,
        module=module,
        description=description,
    )

    return permission


async def get_permission(permission_id: UUID) -> Permission:
    """
    获取权限详情

    Args:
        permission_id: 权限ID

    Returns:
        Permission: 权限实例

    Raises:
        ResourceNotFoundError: 当权限不存在时抛出
    """
    try:
        permission = await Permission.get(id=permission_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"权限 {permission_id} 不存在")

    return permission


async def list_permissions(
    module: Optional[str] = None,
    offset: int = 0,
    limit: int = 10,
) -> List[Permission]:
    """
    获取权限列表

    Args:
        module: 可选的模块名称过滤
        offset: 分页偏移量
        limit: 分页大小

    Returns:
        List[Permission]: 权限列表
    """
    query = Permission.all()

    if module is not None:
        query = query.filter(module=module)

    # 按模块和标识符排序
    query = query.order_by("module", "identifier")

    permissions = await query.offset(offset).limit(limit)
    return permissions


async def _validate_permission_identifier(permission: Permission, new_identifier: str) -> None:
    """验证权限标识是否可用"""
    if new_identifier != permission.identifier:
        existing = await Permission.filter(identifier=new_identifier).first()
        if existing:
            raise ResourceConflictError(f"权限标识 '{new_identifier}' 已存在")


async def update_permission(
    permission_id: UUID,
    name: Optional[str] = None,
    identifier: Optional[str] = None,
    module: Optional[str] = None,
    description: Optional[str] = None,
) -> Permission:
    """
    更新权限信息

    Args:
        permission_id: 权限ID
        name: 新的权限名称
        identifier: 新的权限标识
        module: 新的所属模块
        description: 新的权限描述

    Returns:
        Permission: 更新后的权限实例

    Raises:
        ResourceNotFoundError: 当权限不存在时抛出
        ResourceConflictError: 当新权限标识已存在时抛出
    """
    try:
        permission = await Permission.get(id=permission_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"权限 {permission_id} 不存在")

    # 验证权限标识
    if identifier is not None:
        await _validate_permission_identifier(permission, identifier)
        permission.identifier = identifier

    # 更新其他字段
    if name is not None:
        permission.name = name
    if module is not None:
        permission.module = module
    if description is not None:
        permission.description = description

    await permission.save()
    return permission


@atomic()
async def delete_permission(permission_id: UUID) -> None:
    """
    删除权限

    Args:
        permission_id: 权限ID

    Raises:
        ResourceNotFoundError: 当权限不存在时抛出
    """
    try:
        permission = await Permission.get(id=permission_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"权限 {permission_id} 不存在")

    # 删除权限（这里可能需要处理关联的角色权限关系）
    await permission.delete()


async def count_permissions(
    module: Optional[str] = None,
) -> int:
    """
    获取权限总数

    Args:
        module: 可选的模块名称过滤

    Returns:
        int: 权限总数
    """
    query = Permission.all()

    if module is not None:
        query = query.filter(module=module)

    return await query.count()
