from typing import List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.system.models import Menu
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_menu(
    name: str,
    path: str,
    component: str,
    locale: Optional[str] = None,
    icon: Optional[str] = None,
    requires_auth: bool = False,
    order: int = 0,
    hide_in_menu: bool = False,
    hide_children_in_menu: bool = False,
    no_affix: bool = False,
    ignore_cache: bool = False,
    parent_id: Optional[UUID] = None,
) -> Menu:
    """
    创建新菜单

    Args:
        name: 菜单名称
        path: 菜单路径
        component: 组件路径
        locale: 国际化
        icon: 图标
        requires_auth: 是否需要权限
        order: 排序
        hide_in_menu: 是否隐藏在菜单中
        hide_children_in_menu: 是否隐藏子菜单
        no_affix: 是否不固定在标签栏
        ignore_cache: 是否不缓存
        parent_id: 父菜单ID

    Returns:
        Menu: 创建的菜单实例

    Raises:
        ResourceConflictError: 当菜单名称已存在时抛出
        ResourceNotFoundError: 当父菜单不存在时抛出
    """
    # 检查菜单名称是否已存在（同一父菜单下）
    existing = await Menu.filter(name=name, parent_id=parent_id).first()
    if existing:
        raise ResourceConflictError(f"同级菜单下已存在名称为 '{name}' 的菜单")

    # 如果指定了父菜单，检查父菜单是否存在
    if parent_id:
        parent = await Menu.filter(id=parent_id).first()
        if not parent:
            raise ResourceNotFoundError(f"父菜单 {parent_id} 不存在")

    # 创建新菜单
    menu = await Menu.create(
        name=name,
        path=path,
        component=component,
        locale=locale,
        icon=icon,
        requires_auth=requires_auth,
        order=order,
        hide_in_menu=hide_in_menu,
        hide_children_in_menu=hide_children_in_menu,
        no_affix=no_affix,
        ignore_cache=ignore_cache,
        parent_id=parent_id,
    )

    return menu


async def get_menu(menu_id: UUID) -> Menu:
    """
    获取菜单详情

    Args:
        menu_id: 菜单ID

    Returns:
        Menu: 菜单实例

    Raises:
        ResourceNotFoundError: 当菜单不存在时抛出
    """
    try:
        menu = await Menu.get(id=menu_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"菜单 {menu_id} 不存在")

    return menu


async def list_menus(
    parent_id: Optional[UUID] = None,
    offset: int = 0,
    limit: int = 10,
) -> List[Menu]:
    """
    获取菜单列表

    Args:
        parent_id: 可选的父菜单ID过滤
        offset: 分页偏移量
        limit: 分页大小

    Returns:
        List[Menu]: 菜单列表
    """
    query = Menu.all()

    if parent_id is not None:
        query = query.filter(parent_id=parent_id)

    # 按排序顺序和名称排序
    query = query.order_by("order", "name")

    menus = await query.offset(offset).limit(limit)
    return menus


async def _validate_menu_name(menu: Menu, new_name: str, new_parent_id: Optional[UUID]) -> None:
    """验证菜单名称是否可用"""
    if new_name != menu.name or new_parent_id != menu.parent_id:
        existing = await Menu.filter(name=new_name, parent_id=new_parent_id).first()
        if existing:
            raise ResourceConflictError(f"同级菜单下已存在名称为 '{new_name}' 的菜单")


async def _validate_parent_menu(menu: Menu, new_parent_id: UUID) -> None:
    """验证父菜单是否存在且不会形成循环"""
    if new_parent_id == menu.id:
        raise ResourceConflictError("不能将菜单设置为自己的子菜单")

    parent = await Menu.filter(id=new_parent_id).first()
    if not parent:
        raise ResourceNotFoundError(f"父菜单 {new_parent_id} 不存在")

    # 检查是否会形成循环
    current = parent
    while current.parent_id:
        if current.parent_id == menu.id:
            raise ResourceConflictError("不能将菜单设置为其子菜单的子菜单")
        current = await Menu.get(id=current.parent_id)


async def _update_menu_name_and_parent(
    menu: Menu,
    name: Optional[str],
    parent_id: Optional[UUID],
) -> None:
    """更新菜单名称和父菜单"""
    if name is not None or parent_id is not None:
        new_name = name if name is not None else menu.name
        new_parent_id = parent_id if parent_id is not None else menu.parent_id

        # 验证名称和父菜单
        await _validate_menu_name(menu, new_name, new_parent_id)
        if parent_id is not None:
            await _validate_parent_menu(menu, parent_id)

        # 更新字段
        menu.name = new_name
        menu.parent_id = new_parent_id


async def _update_menu_basic_fields(
    menu: Menu,
    path: Optional[str],
    component: Optional[str],
    locale: Optional[str],
    icon: Optional[str],
    requires_auth: Optional[bool],
    order: Optional[int],
    hide_in_menu: Optional[bool],
    hide_children_in_menu: Optional[bool],
    no_affix: Optional[bool],
    ignore_cache: Optional[bool],
) -> None:
    """更新菜单基本字段"""
    # 创建一个包含所有非空字段的更新字典
    updates = {
        k: v
        for k, v in {
            "path": path,
            "component": component,
            "locale": locale,
            "icon": icon,
            "requires_auth": requires_auth,
            "order": order,
            "hide_in_menu": hide_in_menu,
            "hide_children_in_menu": hide_children_in_menu,
            "no_affix": no_affix,
            "ignore_cache": ignore_cache,
        }.items()
        if v is not None
    }

    # 批量更新字段
    for field, value in updates.items():
        setattr(menu, field, value)


async def _get_menu_for_update(menu_id: UUID) -> Menu:
    """获取要更新的菜单"""
    try:
        return await Menu.get(id=menu_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"菜单 {menu_id} 不存在")


async def update_menu(
    menu_id: UUID,
    name: Optional[str] = None,
    path: Optional[str] = None,
    component: Optional[str] = None,
    locale: Optional[str] = None,
    icon: Optional[str] = None,
    requires_auth: Optional[bool] = None,
    order: Optional[int] = None,
    hide_in_menu: Optional[bool] = None,
    hide_children_in_menu: Optional[bool] = None,
    no_affix: Optional[bool] = None,
    ignore_cache: Optional[bool] = None,
    parent_id: Optional[UUID] = None,
) -> Menu:
    """
    更新菜单信息

    Args:
        menu_id: 菜单ID
        name: 新的菜单名称
        path: 新的菜单路径
        component: 新的组件路径
        locale: 新的国际化
        icon: 新的图标
        requires_auth: 是否需要权限
        order: 新的排序
        hide_in_menu: 是否隐藏在菜单中
        hide_children_in_menu: 是否隐藏子菜单
        no_affix: 是否不固定在标签栏
        ignore_cache: 是否不缓存
        parent_id: 新的父菜单ID

    Returns:
        Menu: 更新后的菜单实例

    Raises:
        ResourceNotFoundError: 当菜单不存在时抛出
        ResourceConflictError: 当新菜单名称已存在或父菜单设置不合法时抛出
    """
    # 获取要更新的菜单
    menu = await _get_menu_for_update(menu_id)

    # 更新名称和父菜单（如果有变更）
    await _update_menu_name_and_parent(menu, name, parent_id)

    # 更新基本字段
    await _update_menu_basic_fields(
        menu,
        path=path,
        component=component,
        locale=locale,
        icon=icon,
        requires_auth=requires_auth,
        order=order,
        hide_in_menu=hide_in_menu,
        hide_children_in_menu=hide_children_in_menu,
        no_affix=no_affix,
        ignore_cache=ignore_cache,
    )

    # 保存更新
    await menu.save()
    return menu


async def get_menu_tree() -> List[Menu]:
    """
    获取菜单树形结构

    Returns:
        List[Menu]: 树形结构的菜单列表
    """
    # 获取所有菜单并按顺序排序
    menus = await Menu.all().order_by("order", "name")

    # 构建菜单树
    menu_dict = {}
    root_menus = []

    # 初始化所有菜单的 children 列表
    for menu in menus:
        menu_dict[menu.id] = menu
        menu.children = []  # 初始化空的子菜单列表
        # 确保所有必需的字段都存在
        if not hasattr(menu, "locale"):
            menu.locale = None
        if not hasattr(menu, "redirect"):
            menu.redirect = None
        if not hasattr(menu, "requires_auth"):
            menu.requires_auth = False
        if not hasattr(menu, "hide_children_in_menu"):
            menu.hide_children_in_menu = False
        if not hasattr(menu, "no_affix"):
            menu.no_affix = False

    # 构建树形结构
    for menu in menus:
        if menu.parent_id is None:
            root_menus.append(menu)
        else:
            parent = menu_dict.get(menu.parent_id)
            if parent:
                parent.children.append(menu)

    return root_menus


@atomic()
async def delete_menu(menu_id: UUID) -> None:
    """删除菜单

    Args:
        menu_id (UUID): 菜单ID

    Raises:
        ResourceNotFoundError: 菜单不存在时抛出此错误
    """
    menu = await Menu.get_or_none(id=menu_id)
    if not menu:
        raise ResourceNotFoundError("菜单不存在")

    # 递归删除所有子菜单
    children = await Menu.filter(parent_id=menu_id)
    for child in children:
        await delete_menu(child.id)

    await menu.delete()


async def count_menus(
    parent_id: Optional[UUID] = None,
) -> int:
    """
    获取菜单总数

    Args:
        parent_id: 可选的父菜单ID过滤

    Returns:
        int: 菜单总数
    """
    query = Menu.all()

    if parent_id is not None:
        query = query.filter(parent_id=parent_id)

    return await query.count()
