from typing import List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.system.models import File
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_file(
    name: str,
    path: str,
    size: int,
    type: str,
    description: Optional[str] = None,
    status: int = 1,
) -> File:
    existing = await File.filter(name=name, path=path).first()
    if existing:
        raise ResourceConflictError(f"文件名 '{name}' 已存在于该路径")
    file_obj = await File.create(
        name=name,
        path=path,
        size=size,
        type=type,
        description=description,
        status=status,
    )
    return file_obj


async def get_file(file_id: UUID) -> File:
    try:
        file_obj = await File.get(id=file_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"文件 {file_id} 不存在")
    return file_obj


async def list_files(status: Optional[int] = None, offset: int = 0, limit: int = 10) -> List[File]:
    query = File.all()
    if status is not None:
        query = query.filter(status=status)
    files = await query.offset(offset).limit(limit)
    return files


async def update_file(
    file_id: UUID,
    name: Optional[str] = None,
    description: Optional[str] = None,
    status: Optional[int] = None,
    path: Optional[str] = None,
    size: Optional[int] = None,
    type: Optional[str] = None,
) -> File:
    try:
        file_obj = await File.get(id=file_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"文件 {file_id} 不存在")
    if name is not None and name != file_obj.name:
        existing = await File.filter(name=name, path=file_obj.path).first()
        if existing:
            raise ResourceConflictError(f"文件名 '{name}' 已存在于该路径")
        file_obj.name = name
    if description is not None:
        file_obj.description = description
    if status is not None:
        file_obj.status = status
    if path is not None:
        file_obj.path = path
    if size is not None:
        file_obj.size = size
    if type is not None:
        file_obj.type = type
    await file_obj.save()
    return file_obj


@atomic()
async def delete_file(file_id: UUID) -> None:
    try:
        file_obj = await File.get(id=file_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"文件 {file_id} 不存在")
    await file_obj.delete()


async def count_files(status: Optional[int] = None) -> int:
    query = File.all()
    if status is not None:
        query = query.filter(status=status)
    return await query.count()
