from typing import Dict, List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.system.models import Config
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_config(
    name: str,
    value: Dict,
    type: str,
    description: Optional[str] = None,
) -> Config:
    """
    创建新配置

    Args:
        name: 配置名称
        value: 配置值
        type: 配置类型
        description: 配置描述

    Returns:
        Config: 创建的配置实例

    Raises:
        ResourceConflictError: 当配置名称已存在时抛出
    """
    # 检查配置名称是否已存在
    existing = await Config.filter(name=name).first()
    if existing:
        raise ResourceConflictError(f"配置名称 '{name}' 已存在")

    # 创建新配置
    config = await Config.create(
        name=name,
        description=description,
        value=value,
        type=type,
    )

    return config


async def get_config(config_id: UUID) -> Config:
    """
    获取配置详情

    Args:
        config_id: 配置ID

    Returns:
        Config: 配置实例

    Raises:
        ResourceNotFoundError: 当配置不存在时抛出
    """
    try:
        config = await Config.get(id=config_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"配置 {config_id} 不存在")

    return config


async def list_configs(
    type: Optional[str] = None,
    offset: int = 0,
    limit: int = 10,
) -> List[Config]:
    """
    获取配置列表

    Args:
        type: 可选的配置类型过滤
        offset: 分页偏移量
        limit: 分页大小

    Returns:
        List[Config]: 配置列表
    """
    query = Config.all()

    if type is not None:
        query = query.filter(type=type)

    configs = await query.offset(offset).limit(limit)
    return configs


async def _validate_config_name(config: Config, new_name: str) -> None:
    """验证配置名称是否可用"""
    if new_name != config.name:
        existing = await Config.filter(name=new_name).first()
        if existing:
            raise ResourceConflictError(f"配置名称 '{new_name}' 已存在")


async def update_config(
    config_id: UUID,
    name: Optional[str] = None,
    description: Optional[str] = None,
    value: Optional[Dict] = None,
    type: Optional[str] = None,
) -> Config:
    """
    更新配置信息

    Args:
        config_id: 配置ID
        name: 新的配置名称
        description: 新的配置描述
        value: 新的配置值
        type: 新的配置类型

    Returns:
        Config: 更新后的配置实例

    Raises:
        ResourceNotFoundError: 当配置不存在时抛出
        ResourceConflictError: 当新配置名称已存在时抛出
    """
    try:
        config = await Config.get(id=config_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"配置 {config_id} 不存在")

    # 验证名称
    if name is not None:
        await _validate_config_name(config, name)
        config.name = name

    # 更新其他字段
    if description is not None:
        config.description = description
    if value is not None:
        config.value = value
    if type is not None:
        config.type = type

    await config.save()
    return config


@atomic()
async def delete_config(config_id: UUID) -> None:
    """
    删除配置

    Args:
        config_id: 配置ID

    Raises:
        ResourceNotFoundError: 当配置不存在时抛出
    """
    try:
        config = await Config.get(id=config_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"配置 {config_id} 不存在")

    await config.delete()


async def count_configs(
    type: Optional[str] = None,
) -> int:
    """
    获取配置总数

    Args:
        type: 可选的配置类型过滤

    Returns:
        int: 配置总数
    """
    query = Config.all()

    if type is not None:
        query = query.filter(type=type)

    return await query.count()
