import json
import time
import uuid

from fastapi import APIRouter, HTTPException

from algo_flow.app.registry.schemas.node import Heartbeat, NodeInfo, StatusUpdate
from algo_flow.cores.redis import ASYNC_REDIS

router = APIRouter()

# 内存存储节点信息
HEARTBEAT_TIMEOUT = 30  # 秒
NODE_KEY_PREFIX = "registry:node:"
DEVICE_UID_KEY_PREFIX = "registry:device_uid:"
NODE_EXPIRE = 60  # 节点信息过期时间（秒）


def get_node_key(node_id: str) -> str:
    return f"{NODE_KEY_PREFIX}{node_id}"


def get_device_uid_key(device_uid: str) -> str:
    return f"{DEVICE_UID_KEY_PREFIX}{device_uid}"


@router.post("/register")
async def register(node: NodeInfo):
    # 用 device_uid 查找是否已注册
    device_uid_key = get_device_uid_key(node.device_uid)
    node_id = await ASYNC_REDIS.get(device_uid_key)
    if node_id:
        # 已注册，续期
        await ASYNC_REDIS.expire(device_uid_key, NODE_EXPIRE)
        await ASYNC_REDIS.expire(
            get_node_key(node_id.decode() if isinstance(node_id, bytes) else node_id), NODE_EXPIRE
        )
        return {
            "msg": "already registered",
            "node_id": node_id.decode() if isinstance(node_id, bytes) else node_id,
        }
    # 未注册，分配新 node_id
    node_id = str(uuid.uuid4())
    await ASYNC_REDIS.set(device_uid_key, node_id, ex=NODE_EXPIRE)
    node_data = node.dict()
    node_data["node_id"] = node_id
    await ASYNC_REDIS.hmset(
        get_node_key(node_id),
        {"info": json.dumps(node_data), "last_heartbeat": str(time.time()), "status": "idle"},
    )
    await ASYNC_REDIS.expire(get_node_key(node_id), NODE_EXPIRE)
    return {"msg": "registered", "node_id": node_id}


@router.post("/heartbeat")
async def heartbeat(hb: Heartbeat):
    key = get_node_key(hb.node_id)
    if not await ASYNC_REDIS.exists(key):
        raise HTTPException(status_code=404, detail="Node not registered")
    await ASYNC_REDIS.hset(key, "last_heartbeat", str(time.time()))
    await ASYNC_REDIS.expire(key, NODE_EXPIRE)
    # 续期 device_uid 映射
    node = await ASYNC_REDIS.hget(key, "info")
    node = json.loads(node)
    device_uid_key = get_device_uid_key(node.get("device_uid"))
    if not await ASYNC_REDIS.exists(device_uid_key):
        raise HTTPException(status_code=404, detail="device not exists")
    await ASYNC_REDIS.expire(device_uid_key, NODE_EXPIRE)
    return {"msg": "heartbeat received"}


@router.post("/status")
async def update_status(status: StatusUpdate):
    key = get_node_key(status.node_id)
    if not await ASYNC_REDIS.exists(key):
        raise HTTPException(status_code=404, detail="Node not registered")
    await ASYNC_REDIS.hset(key, "status", status.status)
    await ASYNC_REDIS.expire(key, NODE_EXPIRE)
    return {"msg": "status updated"}


@router.get("")
async def list_nodes():
    now = time.time()
    result = []
    async for key in ASYNC_REDIS.scan_iter(f"{NODE_KEY_PREFIX}*"):
        data = await ASYNC_REDIS.hgetall(key)
        node_id = key.decode().split(":")[-1] if isinstance(key, bytes) else key.split(":")[-1]
        info = json.loads(data.get("info", "{}"))
        status = data.get("status", "unknown")
        last_heartbeat = float(data.get("last_heartbeat", "0"))
        online = (now - last_heartbeat) < HEARTBEAT_TIMEOUT
        result.append({"node_id": node_id, "info": info, "status": status, "online": online})
    return result
