from typing import Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security, status

from algo_flow.app.media.crud.resource import (
    count_videos,
    create_video,
    delete_video,
    get_video,
    list_videos,
    update_video,
)
from algo_flow.app.media.schemas.resource import (
    VideoCreate,
    VideoList,
    VideoPlayUrl,
    VideoResponse,
    VideoUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.config import settings
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

router = APIRouter()


def _build_play_urls(video) -> VideoPlayUrl:
    zlmk = settings.zlmediakit
    return VideoPlayUrl(
        rtsp=f"rtsp://{zlmk.host}:{zlmk.rtsp_port}/live/{video.id}",
        rtmp=f"rtmp://{zlmk.host}:{zlmk.rtmp_port}/live/{video.id}",
        http_flv=f"//{zlmk.host}:{zlmk.http_port}/live/{video.id}.live.flv",
        hls=f"//{zlmk.host}:{zlmk.http_port}/live/{video.id}/hls.m3u8",
        fmp4=f"//{zlmk.host}:{zlmk.http_port}/live/{video.id}/hls.fmp4.m3u8",
        ts=f"//{zlmk.host}:{zlmk.http_port}/live/{video.id}.live.ts",
        http_mp4=f"//{zlmk.host}:{zlmk.http_port}/live/{video.id}.live.mp4",
    )


@router.post(
    "",
    response_model=VideoResponse,
    status_code=status.HTTP_201_CREATED,
    dependencies=[Security(get_current_active_user, scopes=["media:video:create"])],
    summary="新增视频资源",
)
async def create_video_api(video_in: VideoCreate) -> VideoResponse:
    try:
        db_video = await create_video(
            name=video_in.name,
            type=video_in.type,
            origin_url=video_in.origin_url,
            file_id=video_in.file_id,
            description=video_in.description,
            status=1,
            config=video_in.config,
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))
    return VideoResponse.model_validate(db_video)


@router.get(
    "/{video_id}",
    response_model=VideoResponse,
    dependencies=[Security(get_current_active_user, scopes=["media:video:read"])],
    summary="获取视频详情",
)
async def get_video_api(video_id: UUID = Path(..., description="视频ID")) -> VideoResponse:
    try:
        video = await get_video(video_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    video_dict = video.__dict__.copy()
    video_dict["play_urls"] = _build_play_urls(video)
    return VideoResponse.model_validate(video_dict)


@router.get(
    "",
    response_model=VideoList,
    dependencies=[Security(get_current_active_user, scopes=["media:video:read"])],
    summary="获取视频列表",
)
async def list_videos_api(
    status: Optional[int] = Query(None, description="视频状态过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> VideoList:
    videos = await list_videos(status=status, offset=offset, limit=limit)
    total = await count_videos(status=status)
    items = []
    for v in videos:
        v_dict = v.__dict__.copy()
        v_dict["play_urls"] = _build_play_urls(v)
        items.append(VideoResponse.model_validate(v_dict))
    return VideoList(total=total, items=items)


@router.patch(
    "/{video_id}",
    response_model=VideoResponse,
    dependencies=[Security(get_current_active_user, scopes=["media:video:update"])],
    summary="更新视频",
)
async def update_video_api(
    video_update: VideoUpdate, video_id: UUID = Path(..., description="视频ID")
) -> VideoResponse:
    update_fields = {}
    for field, value in video_update.model_dump(exclude_unset=True).items():
        if value is not None:
            update_fields[field] = value
    try:
        video = await update_video(video_id=video_id, **update_fields)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))
    return VideoResponse.model_validate(video)


@router.delete(
    "/{video_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    dependencies=[Security(get_current_active_user, scopes=["media:video:delete"])],
    summary="删除视频",
)
async def delete_video_api(video_id: UUID = Path(..., description="视频ID")):
    try:
        await delete_video(video_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
