from fastapi import APIRouter, Request
from fastapi.responses import JSONResponse

from algo_flow.app.media.crud.resource import get_video
from algo_flow.app.media.schemas.resource import VideoConfig
from algo_flow.cores.async_http import async_http_request
from algo_flow.cores.config import settings
from algo_flow.cores.log import LOG

router = APIRouter()


@router.post("/on_stream_not_found", summary="ZLMediaKit流未找到回调")
async def on_stream_not_found(request: Request):
    data = await request.json()
    LOG.info(f"[ZLMediaKit] on_stream_not_found: {data}")

    app = data.get("app")
    vhost = data.get("vhost")
    video_id = data.get("stream")

    try:
        video = await get_video(video_id)
    except Exception as e:
        LOG.error(f"未找到视频或ID非法: {e}")
        return JSONResponse({"code": 0, "msg": "not found"})
    # 只对外部流类型做拉流
    if video.type != 2 or not video.origin_url:
        return JSONResponse({"code": 0, "msg": "not external"})

    config = VideoConfig.model_validate(video.config)

    zlmk = settings.zlmediakit
    api_url = f"http://{zlmk.host}:{zlmk.http_port}/index/api/addStreamProxy"
    params = {
        "vhost": vhost,
        "app": app,
        "stream": video_id,
        "url": video.origin_url,
        "secret": zlmk.secret,
        **config.model_dump(exclude_none=True),
    }
    resp = await async_http_request(api_url, method="POST", data=params)
    if not resp or resp.status_code != 200:
        LOG.error("ZLMediaKit拉流失败")
        return JSONResponse({"code": -1, "msg": "拉流失败"})
    resp_json = resp.json()
    code = resp_json.get("code")
    if code != 0:
        msg = resp_json.get("msg")
        LOG.error(f"ZLMediaKit拉流失败: {msg}")
        return JSONResponse({"code": -1, "msg": msg or "拉流失败"})
    return JSONResponse({"code": 0, "msg": "拉流成功"})


@router.post("/stream_none_reader", summary="ZLMediaKit无人观看回调")
async def stream_none_reader(request: Request):
    data = await request.json()
    LOG.info(f"[ZLMediaKit] stream_none_reader: {data}")

    app = data.get("app")
    vhost = data.get("vhost")
    video_id = data.get("stream")

    zlmk = settings.zlmediakit
    api_url = f"http://{zlmk.host}:{zlmk.http_port}/index/api/close_streams"
    params = {
        "vhost": vhost,
        "app": app,
        "stream": video_id,
        "secret": zlmk.secret,
    }
    resp = await async_http_request(api_url, method="POST", data=params)
    if not resp or resp.status_code != 200:
        LOG.error("ZLMediaKit关闭流失败")
        return JSONResponse({"code": -1, "msg": "关闭流失败"})

    resp_json = resp.json()
    code = resp_json.get("code")
    if code != 0:
        msg = resp_json.get("msg")
        LOG.error(f"ZLMediaKit关闭流失败: {msg}")
        return JSONResponse({"code": -1, "msg": msg or "关闭流失败"})

    return JSONResponse({"code": 0, "msg": "关闭流成功"})
