from datetime import datetime
from typing import List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field

from algo_flow.app.media.models import VideoType


class VideoPlayUrl(BaseModel):
    rtmp: Optional[str] = Field(None, description="RTMP播放地址")
    rtsp: Optional[str] = Field(None, description="RTSP播放地址")
    http_flv: Optional[str] = Field(None, description="HTTP-FLV播放地址")
    hls: Optional[str] = Field(None, description="HLS播放地址")
    ts: Optional[str] = Field(None, description="TS播放地址")
    fmp4: Optional[str] = Field(None, description="FMP4播放地址")
    http_mp4: Optional[str] = Field(None, description="MP4播放地址")


class VideoConfig(BaseModel):
    # 你可以根据实际业务扩展字段
    rtp_type: Optional[int] = Field(0, description="拉流方式，0：tcp，1：udp，2：组播")

    enable_hls: Optional[bool] = Field(True, description="是否转换成 hls-mpegts 协议")
    hls_demand: Optional[bool] = Field(True, description="该协议是否有人观看才生成")

    enable_rtsp: Optional[bool] = Field(True, description="是否转 rtsp 协议")
    rtsp_demand: Optional[bool] = Field(True, description="该协议是否有人观看才生成")

    enable_rtmp: Optional[bool] = Field(True, description="是否转 rtmp 协议")
    rtmp_demand: Optional[bool] = Field(True, description="该协议是否有人观看才生成")

    enable_ts: Optional[bool] = Field(False, description="是否转 ts 协议")
    ts_demand: Optional[bool] = Field(True, description="该协议是否有人观看才生成")

    enable_fmp4: Optional[bool] = Field(True, description="是否转 http-fmp4/ws-fmp4 协议")
    fmp4_demand: Optional[bool] = Field(True, description="该协议是否有人观看才生成")

    enable_audio: Optional[bool] = Field(False, description="转协议时是否开启音频")
    add_mute_audio: Optional[bool] = Field(
        False, description="转协议时，无音频是否添加静音 aac 音频"
    )

    enable_mp4: Optional[bool] = Field(False, description="是否允许 mp4 录制")
    mp4_save_path: Optional[str] = Field(None, description="mp4 录制保存路径")
    mp4_max_second: Optional[int] = Field(60 * 60, description="mp4 录制切片大小，单位秒")
    hls_save_path: Optional[str] = Field(None, description="hls 录制保存路径")
    modify_stamp: Optional[int] = Field(
        0, description="该流是否开启时间戳覆盖(0:绝对时间戳/1:系统时间戳/2:相对时间戳)"
    )

    auto_close: Optional[bool] = Field(
        False, description="无人观看是否自动关闭流(不触发无人观看 hook)"
    )


class VideoBase(BaseModel):
    name: str = Field(..., description="视频名称", min_length=1, max_length=255)
    type: VideoType = Field(..., description="视频类型：1-上传，2-外部流")
    origin_url: Optional[str] = Field(None, description="原始外部流地址")
    file_id: Optional[UUID] = Field(None, description="上传视频文件ID")
    description: Optional[str] = Field(None, description="视频描述")
    config: Optional[VideoConfig] = Field(
        default_factory=VideoConfig, description="ZLMediaKit相关配置"
    )
    play_urls: Optional[VideoPlayUrl] = Field(
        default_factory=VideoPlayUrl, description="可播放的流地址列表"
    )


class VideoCreate(VideoBase):
    pass


class VideoUpdate(BaseModel):
    name: Optional[str] = Field(None, description="视频名称", min_length=1, max_length=255)
    type: Optional[VideoType] = Field(None, description="视频类型")
    origin_url: Optional[str] = Field(None, description="原始外部流地址")
    file_id: Optional[UUID] = Field(None, description="上传视频文件ID")
    description: Optional[str] = Field(None, description="视频描述")
    status: Optional[int] = Field(None, description="视频状态")
    config: Optional[VideoConfig] = Field(
        default_factory=VideoConfig, description="ZLMediaKit相关配置"
    )


class VideoResponse(VideoBase):
    id: UUID = Field(..., description="视频ID")
    status: int = Field(..., description="视频状态")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")
    model_config = ConfigDict(from_attributes=True)


class VideoList(BaseModel):
    total: int = Field(..., description="总数")
    items: List[VideoResponse] = Field(..., description="视频列表")
