from typing import List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.media.models import Resource, VideoType
from algo_flow.app.system.models import File
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_video(
    name: str,
    type: VideoType,
    origin_url: Optional[str] = None,
    file_id: Optional[UUID] = None,
    description: Optional[str] = None,
    status: int = 1,
    config: Optional[dict] = None,
) -> Resource:
    existing = await Resource.filter(name=name).first()
    if existing:
        raise ResourceConflictError(f"视频名称 '{name}' 已存在")
    file = await File.get_or_none(id=file_id) if file_id else None
    video = await Resource.create(
        name=name,
        type=type,
        origin_url=origin_url,
        file=file,
        description=description,
        status=status,
        config=config,
    )
    return video


async def get_video(video_id: UUID) -> Resource:
    try:
        video = await Resource.get(id=video_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"视频 {video_id} 不存在")
    return video


async def list_videos(
    status: Optional[int] = None, offset: int = 0, limit: int = 10
) -> List[Resource]:
    query = Resource.all()
    if status is not None:
        query = query.filter(status=status)
    videos = await query.offset(offset).limit(limit)
    return videos


def _update_video_fields(
    video: Resource,
    type: Optional[VideoType] = None,
    origin_url: Optional[str] = None,
    file_id: Optional[UUID] = None,
    description: Optional[str] = None,
    status: Optional[int] = None,
    config: Optional[dict] = None,
) -> None:
    if type is not None:
        video.type = type
    if origin_url is not None:
        video.origin_url = origin_url
    if file_id is not None:
        # File is async, so this will be handled in update_video
        pass
    if description is not None:
        video.description = description
    if status is not None:
        video.status = status
    if config is not None:
        video.config = config


async def update_video(
    video_id: UUID,
    name: Optional[str] = None,
    type: Optional[VideoType] = None,
    origin_url: Optional[str] = None,
    file_id: Optional[UUID] = None,
    description: Optional[str] = None,
    status: Optional[int] = None,
    config: Optional[dict] = None,
) -> Resource:
    try:
        video = await Resource.get(id=video_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"视频 {video_id} 不存在")
    if name is not None and name != video.name:
        existing = await Resource.filter(name=name).first()
        if existing:
            raise ResourceConflictError(f"视频名称 '{name}' 已存在")
        video.name = name
    # Handle file_id separately due to async
    if file_id is not None:
        file = await File.get_or_none(id=file_id)
        video.file = file
    # Update other fields
    _update_video_fields(
        video,
        type=type,
        origin_url=origin_url,
        description=description,
        status=status,
        config=config,
    )
    await video.save()
    return video


@atomic()
async def delete_video(video_id: UUID) -> None:
    try:
        video = await Resource.get(id=video_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"视频 {video_id} 不存在")
    await video.delete()


async def count_videos(status: Optional[int] = None) -> int:
    query = Resource.all()
    if status is not None:
        query = query.filter(status=status)
    return await query.count()
