import contextlib
import importlib
import pkgutil
from typing import AsyncIterator

from fastapi import FastAPI, Request
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse
from ghkit.log import setup_logging
from starlette.status import HTTP_500_INTERNAL_SERVER_ERROR
from tortoise.contrib.fastapi import register_tortoise

from algo_flow.app.system.crud.user import create_user
from algo_flow.cores.config import settings
from algo_flow.cores.exceptions import ResourceConflictError
from algo_flow.cores.log import LOG
from algo_flow.cores.messager import MESSAGE_FACTORY
from algo_flow.cores.model import TORTOISE_ORM, close_db, init_db
from algo_flow.cores.nacos_client import deregister_instance, heartbeat, register_instance
from algo_flow.cores.scope import init_scopes
from algo_flow.cores.sio import attach_socketio


def register_routes(_app: FastAPI):
    LOG.info("Scanning and registering routes...")

    # 扫描 `app` 目录下所有子模块（假设 `app` 目录下的 `urls.py` 里有 `router`）
    package_name = "algo_flow.app"
    for finder, name, is_pkg in pkgutil.iter_modules([package_name.replace(".", "/")]):
        if not is_pkg:
            continue
        module_name = f"{package_name}.{name}.urls"
        try:
            module = importlib.import_module(module_name)
            if hasattr(module, "router"):
                _app.include_router(module.router, prefix=settings.app.api_version)
                LOG.debug(f"Registered router from {module_name}")
        except ModuleNotFoundError as ex:
            LOG.exception(ex)
            LOG.warning(f"No URLs found in {module_name}, skipping...")

    LOG.info("All routes registered.")


@contextlib.asynccontextmanager
async def lifespan(_app: FastAPI) -> AsyncIterator[None]:
    LOG.info("Starting application lifespan...")
    # 应用启动时的初始化
    await init_db()

    try:
        # 创建超级管理员
        await create_user(
            username="superadmin", password="admin123456", email="admin@123.com", is_superuser=True
        )
    except ResourceConflictError:
        pass

    # 在这里注册 Tortoise，以确保在路由中使用 Tortoise 之前数据库已经初始化
    register_tortoise(
        _app,
        config=TORTOISE_ORM,
        generate_schemas=False,
        add_exception_handlers=True,
    )

    # 初始化全局的 scopes
    await init_scopes()

    # 注册路由
    register_routes(_app)

    # 注册 Socket.IO
    attach_socketio(_app)

    # nacos注册服务
    try:
        register_instance()
        heartbeat()
        LOG.info("已注册到Nacos服务中心，并启动心跳线程")
    except Exception as e:
        LOG.exception(e)
        LOG.error(f"注册到Nacos失败: {e}")

    # 通过 yield 将控制权交给 FastAPI
    yield

    # 应用关闭时的清理
    await close_db()

    try:
        deregister_instance()
        LOG.info("已从Nacos注销服务")
    except Exception as e:
        LOG.error(f"从Nacos注销失败: {e}")


async def global_exception_handler(request: Request, exc: Exception):
    # 发送消息通知
    await MESSAGE_FACTORY.async_send_alarm(
        message={
            "config": {"update_multi": True},
            "card_link": {"url": ""},
            "i18n_elements": {
                "zh_cn": [
                    {
                        "tag": "markdown",
                        "content": f" ** URL **: {request.url}",
                        "text_align": "left",
                        "text_size": "normal",
                        "icon": {
                            "tag": "standard_icon",
                            "token": "computer_outlined",
                            "color": "grey",
                        },
                    },
                    {
                        "tag": "markdown",
                        "content": f" ** METHOD **: {request.method}",
                        "text_align": "left",
                        "text_size": "normal",
                        "icon": {
                            "tag": "standard_icon",
                            "token": "lan_outlined",
                            "color": "grey",
                        },
                    },
                    {
                        "tag": "markdown",
                        "content": f" ** STACK **: {exc}",
                        "text_align": "left",
                        "text_size": "normal",
                        "icon": {
                            "tag": "standard_icon",
                            "token": "ram_outlined",
                            "color": "grey",
                        },
                    },
                ]
            },
            "i18n_header": {
                "zh_cn": {
                    "title": {"tag": "plain_text", "content": "接口异常"},
                    "subtitle": {"tag": "plain_text", "content": ""},
                    "template": "red",
                    "ud_icon": {"tag": "standard_icon", "token": "alarm_outlined"},
                }
            },
        }
    )
    # 这里可以根据需要自定义错误响应
    return JSONResponse(
        status_code=HTTP_500_INTERNAL_SERVER_ERROR,
        content={"code": 500, "msg": "服务器内部错误", "data": None},
    )


def make_app(lifespan_func=None):
    setup_logging(level=settings.app.logging_level)  # 启用 loguru 作为全局日志系统

    # 使用传入的 lifespan 函数，如果没有传入则使用默认的 lifespan
    lifespan_to_use = lifespan_func if lifespan_func is not None else lifespan

    app = FastAPI(
        title=settings.app.project_name,
        debug=settings.app.debug,
        lifespan=lifespan_to_use,
        docs_url=settings.app.doc_path,
        redoc_url=None,
        openapi_url=f"{settings.app.doc_path}.json",
    )
    # 添加 CORS 中间件
    app.add_middleware(
        CORSMiddleware,
        allow_origins=["*"],  # 允许所有来源
        allow_credentials=True,
        allow_methods=["*"],  # 允许所有方法
        allow_headers=["*"],  # 允许所有头部
    )
    # 注册全局异常处理器
    app.add_exception_handler(Exception, global_exception_handler)

    return app
