import hashlib
import hmac
import subprocess

from fastapi import APIRouter, Header, HTTPException, Request

from algo_flow.cores.config import settings
from algo_flow.cores.log import LOG
from algo_flow.cores.nacos_client import NACOS_CLIENT

common_router = APIRouter()


@common_router.post(
    "/healthy",
    summary="healthy",
)
async def healthy():
    return None


@common_router.post("/github-webhook")
async def github_webhook(request: Request, x_hub_signature_256: str = Header(None)):
    """
    GitHub Webhook
    功能：接收 GitHub Webhook 通知，自动拉取最新代码
    :param request:
    :param x_hub_signature_256:
    :return:
    """
    # 获取请求体
    body = await request.body()
    # 计算签名
    signature = (
        "sha256="
        + hmac.new(settings.github.webhook_secret.encode(), body, hashlib.sha256).hexdigest()
    )

    # 比较签名
    if not hmac.compare_digest(signature, x_hub_signature_256):
        raise HTTPException(status_code=400, detail="Invalid signature")

    # 拉取最新代码
    subprocess.run(["git", "pull"], check=True)

    return {"status": "success", "message": "Code pulled successfully"}


@common_router.get("/nacos/services", summary="查询nacos服务列表")
def nacos_services():
    try:
        services = NACOS_CLIENT.list_naming_instance(
            service_name=f"{settings.app.service_name}-{settings.app.environment}"
        )
        LOG.info(f"查询到的服务列表: {services}")
        return services
    except Exception as e:
        return {"error": str(e)}
