from typing import Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security
from starlette.status import (
    HTTP_201_CREATED,
    HTTP_204_NO_CONTENT,
    HTTP_400_BAD_REQUEST,
    HTTP_404_NOT_FOUND,
)

from algo_flow.app.algo.crud.project import (
    count_projects,
    create_project,
    delete_project,
    get_project,
    list_projects,
    update_project,
)
from algo_flow.app.algo.schemas.project import (
    ProjectCreate,
    ProjectList,
    ProjectResponse,
    ProjectUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import ProjectStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

router = APIRouter()


@router.post(
    "",
    response_model=ProjectResponse,
    status_code=HTTP_201_CREATED,
    summary="创建项目",
    dependencies=[Security(get_current_active_user, scopes=["algo:project:create"])],
)
async def create_project_api(project: ProjectCreate) -> ProjectResponse:
    """
    创建新项目

    Args:
        project: 项目创建参数

    Returns:
        ProjectResponse: 创建的项目信息

    Raises:
        HTTPException: 当项目名称已存在时抛出 400 错误
    """
    try:
        db_project = await create_project(name=project.name, description=project.description)
    except ResourceConflictError as e:
        raise HTTPException(status_code=HTTP_400_BAD_REQUEST, detail=str(e))

    return ProjectResponse.model_validate(db_project)


@router.get(
    "/{project_id}",
    response_model=ProjectResponse,
    summary="获取项目详情",
    dependencies=[Security(get_current_active_user, scopes=["algo:project:read"])],
)
async def get_project_api(project_id: UUID = Path(..., description="项目ID")) -> ProjectResponse:
    """
    获取项目详情

    Args:
        project_id: 项目ID

    Returns:
        ProjectResponse: 项目详细信息

    Raises:
        HTTPException: 当项目不存在时抛出 404 错误
    """
    try:
        project = await get_project(project_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=HTTP_404_NOT_FOUND, detail=str(e))

    return ProjectResponse.model_validate(project)


@router.get(
    "",
    response_model=ProjectList,
    summary="获取项目列表",
    dependencies=[Security(get_current_active_user, scopes=["algo:project:read"])],
)
async def list_projects_api(
    status: Optional[ProjectStatus] = Query(None, description="项目状态过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> ProjectList:
    """
    获取项目列表

    Args:
        status: 可选的项目状态过滤
        offset: 分页偏移量，默认0
        limit: 分页大小，默认10，最大100

    Returns:
        ProjectList: 项目列表及总数
    """
    projects = await list_projects(status=status, offset=offset, limit=limit)
    total = await count_projects(status=status)

    return ProjectList(
        total=total,
        items=[ProjectResponse.model_validate(p) for p in projects],
    )


@router.patch(
    "/{project_id}",
    response_model=ProjectResponse,
    summary="更新项目",
    dependencies=[Security(get_current_active_user, scopes=["algo:project:update"])],
)
async def update_project_api(
    project_update: ProjectUpdate, project_id: UUID = Path(..., description="项目ID")
) -> ProjectResponse:
    """
    更新项目信息

    Args:
        project_id: 项目ID
        project_update: 要更新的项目信息

    Returns:
        ProjectResponse: 更新后的项目信息

    Raises:
        HTTPException: 当项目不存在时抛出 404 错误，当项目名称已存在时抛出 400 错误
    """
    try:
        project = await update_project(
            project_id=project_id,
            name=project_update.name,
            description=project_update.description,
            status=project_update.status,
        )
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=HTTP_400_BAD_REQUEST, detail=str(e))

    return ProjectResponse.model_validate(project)


@router.delete(
    "/{project_id}",
    status_code=HTTP_204_NO_CONTENT,
    summary="删除项目",
    dependencies=[Security(get_current_active_user, scopes=["algo:project:delete"])],
)
async def delete_project_api(project_id: UUID = Path(..., description="项目ID")):
    """
    删除项目

    Args:
        project_id: 项目ID

    Raises:
        HTTPException: 当项目不存在时抛出 404 错误
    """
    try:
        await delete_project(project_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=HTTP_404_NOT_FOUND, detail=str(e))
