from typing import Optional
from uuid import UUID

from fastapi import APIRouter, Path, Query, Security
from starlette.status import (
    HTTP_201_CREATED,
    HTTP_204_NO_CONTENT,
)

from algo_flow.app.algo.crud.pipeline_step import (
    count_pipeline_steps,
    create_pipeline_step,
    delete_pipeline_step,
    get_pipeline_step,
    list_pipeline_steps,
    update_pipeline_step,
)
from algo_flow.app.algo.schemas.pipeline_step import (
    PipelineStepCreate,
    PipelineStepList,
    PipelineStepResponse,
    PipelineStepUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import PipelineStepType, TaskStatus

router = APIRouter()


@router.post(
    "",
    response_model=PipelineStepResponse,
    status_code=HTTP_201_CREATED,
    summary="创建管道步骤",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline_step:create"])],
)
async def create_pipeline_step_api(step: PipelineStepCreate) -> PipelineStepResponse:
    db_step = await create_pipeline_step(name=step.name, type=step.type, config=step.config)
    return PipelineStepResponse.model_validate(db_step)


@router.get(
    "/{step_id}",
    response_model=PipelineStepResponse,
    summary="获取管道步骤详情",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline_step:read"])],
)
async def get_pipeline_step_api(
    step_id: UUID = Path(..., description="步骤ID")
) -> PipelineStepResponse:
    step = await get_pipeline_step(step_id)
    return PipelineStepResponse.model_validate(step)


@router.get(
    "",
    response_model=PipelineStepList,
    summary="获取管道步骤列表",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline_step:read"])],
)
async def list_pipeline_steps_api(
    type: Optional[PipelineStepType] = Query(None, description="步骤类型过滤"),
    status: Optional[TaskStatus] = Query(None, description="步骤状态过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> PipelineStepList:
    steps = await list_pipeline_steps(type=type, status=status, offset=offset, limit=limit)
    total = await count_pipeline_steps(type=type, status=status)
    return PipelineStepList(
        total=total,
        items=[PipelineStepResponse.model_validate(s) for s in steps],
    )


@router.patch(
    "/{step_id}",
    response_model=PipelineStepResponse,
    summary="更新管道步骤",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline_step:update"])],
)
async def update_pipeline_step_api(
    step_update: PipelineStepUpdate, step_id: UUID = Path(..., description="步骤ID")
) -> PipelineStepResponse:
    step = await update_pipeline_step(
        step_id=step_id,
        name=step_update.name,
        type=step_update.type,
        config=step_update.config,
        status=step_update.status,
    )
    return PipelineStepResponse.model_validate(step)


@router.delete(
    "/{step_id}",
    status_code=HTTP_204_NO_CONTENT,
    summary="删除管道步骤",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline_step:delete"])],
)
async def delete_pipeline_step_api(step_id: UUID = Path(..., description="步骤ID")):
    await delete_pipeline_step(step_id)
