from typing import Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security
from starlette.status import (
    HTTP_201_CREATED,
    HTTP_204_NO_CONTENT,
    HTTP_400_BAD_REQUEST,
    HTTP_404_NOT_FOUND,
)

from algo_flow.app.algo.crud.pipeline import (
    count_pipelines,
    create_pipeline,
    delete_pipeline,
    get_pipeline,
    list_pipelines,
    update_pipeline,
)
from algo_flow.app.algo.schemas.pipeline import (
    PipelineCreate,
    PipelineList,
    PipelineResponse,
    PipelineUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import TaskStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

router = APIRouter()


@router.post(
    "",
    response_model=PipelineResponse,
    status_code=HTTP_201_CREATED,
    summary="创建管道",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline:create"])],
)
async def create_pipeline_api(pipeline: PipelineCreate) -> PipelineResponse:
    try:
        db_pipeline = await create_pipeline(
            name=pipeline.name, description=pipeline.description, steps=pipeline.steps
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=HTTP_400_BAD_REQUEST, detail=str(e))
    return PipelineResponse.model_validate(db_pipeline)


@router.get(
    "/{pipeline_id}",
    response_model=PipelineResponse,
    summary="获取管道详情",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline:read"])],
)
async def get_pipeline_api(pipeline_id: UUID = Path(..., description="管道ID")) -> PipelineResponse:
    try:
        pipeline = await get_pipeline(pipeline_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=HTTP_404_NOT_FOUND, detail=str(e))
    return PipelineResponse.model_validate(pipeline)


@router.get(
    "",
    response_model=PipelineList,
    summary="获取管道列表",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline:read"])],
)
async def list_pipelines_api(
    status: Optional[TaskStatus] = Query(None, description="管道状态过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> PipelineList:
    pipelines = await list_pipelines(status=status, offset=offset, limit=limit)
    total = await count_pipelines(status=status)
    return PipelineList(
        total=total,
        items=[PipelineResponse.model_validate(p) for p in pipelines],
    )


@router.patch(
    "/{pipeline_id}",
    response_model=PipelineResponse,
    summary="更新管道",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline:update"])],
)
async def update_pipeline_api(
    pipeline_update: PipelineUpdate, pipeline_id: UUID = Path(..., description="管道ID")
) -> PipelineResponse:
    try:
        pipeline = await update_pipeline(
            pipeline_id=pipeline_id,
            name=pipeline_update.name,
            description=pipeline_update.description,
            status=pipeline_update.status,
            steps=pipeline_update.steps,
        )
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=HTTP_400_BAD_REQUEST, detail=str(e))
    return PipelineResponse.model_validate(pipeline)


@router.delete(
    "/{pipeline_id}",
    status_code=HTTP_204_NO_CONTENT,
    summary="删除管道",
    dependencies=[Security(get_current_active_user, scopes=["algo:pipeline:delete"])],
)
async def delete_pipeline_api(pipeline_id: UUID = Path(..., description="管道ID")):
    try:
        await delete_pipeline(pipeline_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=HTTP_404_NOT_FOUND, detail=str(e))
