from typing import Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security, status

from algo_flow.app.algo.crud.model import (
    count_models,
    create_model,
    delete_model,
    get_model,
    list_models,
    update_model,
)
from algo_flow.app.algo.schemas.model import (
    ModelCreate,
    ModelList,
    ModelResponse,
    ModelUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import ModelStatus, ModelType
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

router = APIRouter()


@router.post(
    "",
    response_model=ModelResponse,
    status_code=status.HTTP_201_CREATED,
    summary="创建模型",
    dependencies=[Security(get_current_active_user, scopes=["algo:model:create"])],
)
async def create_model_api(model: ModelCreate) -> ModelResponse:
    """
    创建新模型

    Args:
        model: 模型创建参数

    Returns:
        ModelResponse: 创建的模型信息

    Raises:
        HTTPException: 当项目不存在时抛出 404 错误，当模型名称已存在时抛出 400 错误
    """
    try:
        db_model = await create_model(
            project_id=model.project_id,
            name=model.name,
            description=model.description,
            version=model.version,
            type=model.type,
            format=model.format,
            status=model.status,
            storage_path=model.storage_path,
            config=model.config,
            metrics=model.metrics,
        )
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))

    return ModelResponse.model_validate(db_model)


@router.get(
    "/{model_id}",
    response_model=ModelResponse,
    summary="获取模型详情",
    dependencies=[Security(get_current_active_user, scopes=["algo:model:read"])],
)
async def get_model_api(model_id: UUID = Path(..., description="模型ID")) -> ModelResponse:
    """
    获取模型详情

    Args:
        model_id: 模型ID

    Returns:
        ModelResponse: 模型详细信息

    Raises:
        HTTPException: 当模型不存在时抛出 404 错误
    """
    try:
        model = await get_model(model_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return ModelResponse.model_validate(model)


@router.get(
    "",
    response_model=ModelList,
    summary="获取模型列表",
    dependencies=[Security(get_current_active_user, scopes=["algo:model:read"])],
)
async def list_models_api(
    project_id: Optional[UUID] = Query(None, description="项目ID过滤"),
    status: Optional[ModelStatus] = Query(None, description="模型状态过滤"),
    type: Optional[ModelType] = Query(None, description="模型类型过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> ModelList:
    """
    获取模型列表

    Args:
        project_id: 可选的项目ID过滤
        status: 可选的模型状态过滤
        type: 可选的模型类型过滤
        offset: 分页偏移量，默认0
        limit: 分页大小，默认10，最大100

    Returns:
        ModelList: 模型列表及总数
    """
    models = await list_models(
        project_id=project_id, status=status, type=type, offset=offset, limit=limit
    )
    total = await count_models(project_id=project_id, status=status, type=type)

    return ModelList(total=total, items=[ModelResponse.model_validate(m) for m in models])


@router.patch(
    "/{model_id}",
    response_model=ModelResponse,
    summary="更新模型",
    dependencies=[Security(get_current_active_user, scopes=["algo:model:update"])],
)
async def update_model_api(
    model_update: ModelUpdate, model_id: UUID = Path(..., description="模型ID")
) -> ModelResponse:
    """
    更新模型信息

    Args:
        model_id: 模型ID
        model_update: 要更新的模型信息

    Returns:
        ModelResponse: 更新后的模型信息

    Raises:
        HTTPException: 当模型不存在时抛出 404 错误，当模型名称已存在时抛出 400 错误
    """
    update_fields = {}
    for field, value in model_update.model_dump(exclude_unset=True).items():
        if value is not None:
            update_fields[field] = value

    try:
        model = await update_model(model_id=model_id, **update_fields)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))

    return ModelResponse.model_validate(model)


@router.delete(
    "/{model_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="删除模型",
    dependencies=[Security(get_current_active_user, scopes=["algo:model:delete"])],
)
async def delete_model_api(model_id: UUID = Path(..., description="模型ID")):
    """
    删除模型

    Args:
        model_id: 模型ID

    Raises:
        HTTPException: 当模型不存在时抛出 404 错误
    """
    try:
        await delete_model(model_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
