from typing import Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security, status

from algo_flow.app.algo.crud.dataset import (
    count_datasets,
    create_dataset,
    delete_dataset,
    get_dataset,
    list_datasets,
    update_dataset,
)
from algo_flow.app.algo.schemas.dataset import (
    DatasetCreate,
    DatasetList,
    DatasetResponse,
    DatasetUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import DatasetStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

router = APIRouter()


@router.post(
    "",
    response_model=DatasetResponse,
    status_code=status.HTTP_201_CREATED,
    summary="创建数据集",
    dependencies=[Security(get_current_active_user, scopes=["algo:dataset:create"])],
)
async def create_dataset_api(dataset: DatasetCreate) -> DatasetResponse:
    """
    创建新数据集

    Args:
        dataset: 数据集创建参数

    Returns:
        DatasetResponse: 创建的数据集信息

    Raises:
        HTTPException: 当数据集名称已存在时抛出 400 错误
    """
    try:
        db_dataset = await create_dataset(
            name=dataset.name,
            description=dataset.description,
            status=dataset.status,
            storage_path=dataset.storage_path,
            metadata=dataset.metadata,
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))

    return DatasetResponse.model_validate(db_dataset)


@router.get(
    "/{dataset_id}",
    response_model=DatasetResponse,
    summary="获取数据集详情",
    dependencies=[Security(get_current_active_user, scopes=["algo:dataset:read"])],
)
async def get_dataset_api(dataset_id: UUID = Path(..., description="数据集ID")) -> DatasetResponse:
    """
    获取数据集详情

    Args:
        dataset_id: 数据集ID

    Returns:
        DatasetResponse: 数据集详细信息

    Raises:
        HTTPException: 当数据集不存在时抛出 404 错误
    """
    try:
        dataset = await get_dataset(dataset_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return DatasetResponse.model_validate(dataset)


@router.get(
    "",
    response_model=DatasetList,
    summary="获取数据集列表",
    dependencies=[Security(get_current_active_user, scopes=["algo:dataset:read"])],
)
async def list_datasets_api(
    status: Optional[DatasetStatus] = Query(None, description="数据集状态过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> DatasetList:
    """
    获取数据集列表

    Args:
        status: 可选的数据集状态过滤
        offset: 分页偏移量，默认0
        limit: 分页大小，默认10，最大100

    Returns:
        DatasetList: 数据集列表及总数
    """
    datasets = await list_datasets(status=status, offset=offset, limit=limit)
    total = await count_datasets(status=status)

    return DatasetList(total=total, items=[DatasetResponse.model_validate(d) for d in datasets])


@router.patch(
    "/{dataset_id}",
    response_model=DatasetResponse,
    summary="更新数据集",
    dependencies=[Security(get_current_active_user, scopes=["algo:dataset:update"])],
)
async def update_dataset_api(
    dataset_update: DatasetUpdate, dataset_id: UUID = Path(..., description="数据集ID")
) -> DatasetResponse:
    """
    更新数据集信息

    Args:
        dataset_id: 数据集ID
        dataset_update: 要更新的数据集信息

    Returns:
        DatasetResponse: 更新后的数据集信息

    Raises:
        HTTPException: 当数据集不存在时抛出 404 错误，当数据集名称已存在时抛出 400 错误
    """
    update_fields = {}
    for field, value in dataset_update.model_dump(exclude_unset=True).items():
        if value is not None:
            update_fields[field] = value

    try:
        dataset = await update_dataset(dataset_id=dataset_id, **update_fields)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))

    return DatasetResponse.model_validate(dataset)


@router.delete(
    "/{dataset_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="删除数据集",
    dependencies=[Security(get_current_active_user, scopes=["algo:dataset:delete"])],
)
async def delete_dataset_api(dataset_id: UUID = Path(..., description="数据集ID")):
    """
    删除数据集

    Args:
        dataset_id: 数据集ID

    Raises:
        HTTPException: 当数据集不存在时抛出 404 错误
    """
    try:
        await delete_dataset(dataset_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
