from typing import Optional
from uuid import UUID

from fastapi import APIRouter, HTTPException, Path, Query, Security, status

from algo_flow.app.algo.crud.compute import (
    count_compute_nodes,
    create_compute_node,
    delete_compute_node,
    get_compute_node,
    list_compute_nodes,
    update_compute_node,
)
from algo_flow.app.algo.schemas.compute import (
    ComputeNodeCreate,
    ComputeNodeList,
    ComputeNodeResponse,
    ComputeNodeUpdate,
)
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import ComputeNodeStatus, ComputeNodeType
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

router = APIRouter()


@router.post(
    "",
    response_model=ComputeNodeResponse,
    status_code=status.HTTP_201_CREATED,
    summary="创建计算节点",
    dependencies=[Security(get_current_active_user, scopes=["algo:compute:create"])],
)
async def create_compute_node_api(node: ComputeNodeCreate) -> ComputeNodeResponse:
    """
    创建新计算节点

    Args:
        node: 计算节点创建参数

    Returns:
        ComputeNodeResponse: 创建的计算节点信息

    Raises:
        HTTPException: 当节点名称已存在或IP地址和端口组合已存在时抛出 409 错误
    """
    try:
        db_node = await create_compute_node(
            name=node.name,
            description=node.description,
            type=node.type,
            ip_address=str(node.ip_address),
            port=node.port,
            resources=node.resources,
        )
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return ComputeNodeResponse.model_validate(db_node)


@router.get(
    "/{node_id}",
    response_model=ComputeNodeResponse,
    summary="获取计算节点详情",
    dependencies=[Security(get_current_active_user, scopes=["algo:compute:read"])],
)
async def get_compute_node_api(
    node_id: UUID = Path(..., description="计算节点ID"),
) -> ComputeNodeResponse:
    """
    获取计算节点详情

    Args:
        node_id: 计算节点ID

    Returns:
        ComputeNodeResponse: 计算节点详细信息

    Raises:
        HTTPException: 当计算节点不存在时抛出 404 错误
    """
    try:
        node = await get_compute_node(node_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))

    return ComputeNodeResponse.model_validate(node)


@router.get(
    "",
    response_model=ComputeNodeList,
    summary="获取计算节点列表",
    dependencies=[Security(get_current_active_user, scopes=["algo:compute:read"])],
)
async def list_compute_nodes_api(
    status: Optional[ComputeNodeStatus] = Query(None, description="节点状态过滤"),
    type: Optional[ComputeNodeType] = Query(None, description="节点类型过滤"),
    offset: int = Query(0, ge=0, description="分页偏移量"),
    limit: int = Query(10, ge=1, le=100, description="分页大小"),
) -> ComputeNodeList:
    """
    获取计算节点列表

    Args:
        status: 可选的节点状态过滤
        type: 可选的节点类型过滤
        offset: 分页偏移量，默认0
        limit: 分页大小，默认10，最大100

    Returns:
        ComputeNodeList: 计算节点列表及总数
    """
    nodes = await list_compute_nodes(status=status, type=type, offset=offset, limit=limit)
    total = await count_compute_nodes(status=status, type=type)

    return ComputeNodeList(
        total=total,
        items=[ComputeNodeResponse.model_validate(n) for n in nodes],
    )


@router.patch(
    "/{node_id}",
    response_model=ComputeNodeResponse,
    summary="更新计算节点",
    dependencies=[Security(get_current_active_user, scopes=["algo:compute:update"])],
)
async def update_compute_node_api(
    node_update: ComputeNodeUpdate,
    node_id: UUID = Path(..., description="计算节点ID"),
) -> ComputeNodeResponse:
    """
    更新计算节点信息

    Args:
        node_id: 计算节点ID
        node_update: 要更新的计算节点信息

    Returns:
        ComputeNodeResponse: 更新后的计算节点信息

    Raises:
        HTTPException: 当计算节点不存在时抛出 404 错误，当节点名称已存在或IP地址和端口组合已存在时抛出 409 错误
    """
    try:
        node = await update_compute_node(
            node_id=node_id,
            name=node_update.name,
            description=node_update.description,
            status=node_update.status,
            type=node_update.type,
            ip_address=str(node_update.ip_address) if node_update.ip_address else None,
            port=node_update.port,
            resources=node_update.resources,
        )
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
    except ResourceConflictError as e:
        raise HTTPException(status_code=status.HTTP_409_CONFLICT, detail=str(e))

    return ComputeNodeResponse.model_validate(node)


@router.delete(
    "/{node_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="删除计算节点",
    dependencies=[Security(get_current_active_user, scopes=["algo:compute:delete"])],
)
async def delete_compute_node_api(
    node_id: UUID = Path(..., description="计算节点ID"),
) -> None:
    """
    删除计算节点

    Args:
        node_id: 计算节点ID

    Raises:
        HTTPException: 当计算节点不存在时抛出 404 错误
    """
    try:
        await delete_compute_node(node_id)
    except ResourceNotFoundError as e:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail=str(e))
