from datetime import datetime
from typing import Any, List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field

from algo_flow.cores.constant.algo import PipelineStepType, TaskStatus


class PipelineStepBase(BaseModel):
    """管道步骤基础模型"""

    name: str = Field(..., description="步骤名称", min_length=1, max_length=100)
    type: PipelineStepType = Field(..., description="步骤类型")
    config: Any = Field(default_factory=dict, description="步骤配置")
    status: TaskStatus = Field(default=TaskStatus.PENDING, description="步骤状态")


class PipelineStepCreate(BaseModel):
    """管道步骤创建请求模型"""

    name: str = Field(..., description="步骤名称", min_length=1, max_length=100)
    type: PipelineStepType = Field(..., description="步骤类型")
    config: Any = Field(default_factory=dict, description="步骤配置")


class PipelineStepUpdate(BaseModel):
    """管道步骤更新请求模型"""

    name: Optional[str] = Field(None, description="步骤名称", min_length=1, max_length=100)
    type: Optional[PipelineStepType] = Field(None, description="步骤类型")
    config: Optional[Any] = Field(None, description="步骤配置")
    status: Optional[TaskStatus] = Field(None, description="步骤状态")


class PipelineStepResponse(PipelineStepBase):
    """管道步骤响应模型"""

    id: UUID = Field(..., description="步骤ID")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")
    model_config = ConfigDict(from_attributes=True)


class PipelineStepList(BaseModel):
    """管道步骤列表响应模型"""

    total: int = Field(..., description="总数")
    items: List[PipelineStepResponse] = Field(..., description="步骤列表")
