from datetime import datetime
from typing import Any, List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field

from algo_flow.cores.constant.algo import TaskStatus


class PipelineBase(BaseModel):
    """管道基础模型"""

    name: str = Field(..., description="管道名称", min_length=1, max_length=100)
    description: Optional[str] = Field(None, description="管道描述")
    status: TaskStatus = Field(default=TaskStatus.PENDING, description="管道状态")
    steps: Any = Field(..., description="管道步骤配置")


class PipelineCreate(BaseModel):
    """管道创建请求模型"""

    name: str = Field(..., description="管道名称", min_length=1, max_length=100)
    description: Optional[str] = Field(None, description="管道描述")
    steps: Any = Field(default_factory=dict, description="管道步骤配置")


class PipelineUpdate(BaseModel):
    """管道更新请求模型"""

    name: Optional[str] = Field(None, description="管道名称", min_length=1, max_length=100)
    description: Optional[str] = Field(None, description="管道描述")
    status: Optional[TaskStatus] = Field(None, description="管道状态")
    steps: Optional[Any] = Field(None, description="管道步骤配置")


class PipelineResponse(PipelineBase):
    """管道响应模型"""

    id: UUID = Field(..., description="管道ID")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")
    model_config = ConfigDict(from_attributes=True)


class PipelineList(BaseModel):
    """管道列表响应模型"""

    total: int = Field(..., description="总数")
    items: List[PipelineResponse] = Field(..., description="管道列表")
