from datetime import datetime
from typing import Dict, List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field

from algo_flow.cores.constant.algo import ModelFormat, ModelStatus, ModelType


class ModelBase(BaseModel):
    """模型基础信息"""

    name: str = Field(..., description="模型名称", min_length=1, max_length=100)
    description: Optional[str] = Field(None, description="模型描述")
    version: str = Field(..., description="模型版本号", min_length=1, max_length=50)
    type: ModelType = Field(..., description="模型类型")
    format: ModelFormat = Field(..., description="模型格式")
    status: ModelStatus = Field(default=ModelStatus.DRAFT, description="模型状态")
    storage_path: str = Field(..., description="模型文件存储路径")
    config: Dict = Field(..., description="模型配置信息")
    metrics: Optional[Dict] = Field(None, description="模型评估指标")


class ModelCreate(ModelBase):
    """模型创建请求模型"""

    project_id: UUID = Field(..., description="所属项目ID")


class ModelUpdate(BaseModel):
    """模型更新请求模型"""

    name: Optional[str] = Field(None, description="模型名称", min_length=1, max_length=100)
    description: Optional[str] = Field(None, description="模型描述")
    version: Optional[str] = Field(None, description="模型版本号", min_length=1, max_length=50)
    type: Optional[ModelType] = Field(None, description="模型类型")
    format: Optional[ModelFormat] = Field(None, description="模型格式")
    status: Optional[ModelStatus] = Field(None, description="模型状态")
    storage_path: Optional[str] = Field(None, description="模型文件存储路径")
    config: Optional[Dict] = Field(None, description="模型配置信息")
    metrics: Optional[Dict] = Field(None, description="模型评估指标")


class ModelResponse(ModelBase):
    """模型响应模型"""

    id: UUID = Field(..., description="模型ID")
    project_id: UUID = Field(..., description="所属项目ID")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")

    model_config = ConfigDict(from_attributes=True)


class ModelList(BaseModel):
    """模型列表响应模型"""

    total: int = Field(..., description="总数")
    items: List[ModelResponse] = Field(..., description="模型列表")
