from datetime import datetime
from typing import Dict, List, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field

from algo_flow.cores.constant.algo import DatasetStatus


class DatasetBase(BaseModel):
    """数据集基础信息"""

    name: str = Field(..., description="数据集名称", min_length=1, max_length=100)
    description: Optional[str] = Field(None, description="数据集描述")
    status: DatasetStatus = Field(default=DatasetStatus.PROCESSING, description="数据集状态")
    storage_path: str = Field(..., description="数据集文件存储路径")
    metadata: Dict = Field(..., description="数据集元数据信息")


class DatasetCreate(DatasetBase):
    """数据集创建请求模型"""

    pass


class DatasetUpdate(BaseModel):
    """数据集更新请求模型"""

    name: Optional[str] = Field(None, description="数据集名称", min_length=1, max_length=100)
    description: Optional[str] = Field(None, description="数据集描述")
    status: Optional[DatasetStatus] = Field(None, description="数据集状态")
    storage_path: Optional[str] = Field(None, description="数据集文件存储路径")
    metadata: Optional[Dict] = Field(None, description="数据集元数据信息")


class DatasetResponse(DatasetBase):
    """数据集响应模型"""

    id: UUID = Field(..., description="数据集ID")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")

    model_config = ConfigDict(from_attributes=True)


class DatasetList(BaseModel):
    """数据集列表响应模型"""

    total: int = Field(..., description="总数")
    items: List[DatasetResponse] = Field(..., description="数据集列表")
