from datetime import datetime
from typing import Dict, Optional
from uuid import UUID

from pydantic import BaseModel, ConfigDict, Field, IPvAnyAddress, conint

from algo_flow.cores.constant.algo import ComputeNodeStatus, ComputeNodeType
from algo_flow.cores.schema.base import ListResponse


class ComputeNodeBase(BaseModel):
    """计算节点基础信息"""

    name: str = Field(..., description="节点名称", min_length=1, max_length=100)
    description: Optional[str] = Field(None, description="节点描述")
    type: ComputeNodeType = Field(..., description="节点类型")
    ip_address: IPvAnyAddress = Field(..., description="节点IP地址")
    port: conint(gt=0, lt=65536) = Field(..., description="节点端口")
    resources: Dict = Field(..., description="节点资源信息")


class ComputeNodeCreate(ComputeNodeBase):
    """计算节点创建请求模型"""

    status: ComputeNodeStatus = Field(default=ComputeNodeStatus.OFFLINE, description="节点状态")


class ComputeNodeUpdate(BaseModel):
    """计算节点更新请求模型"""

    name: Optional[str] = Field(None, description="节点名称", min_length=1, max_length=100)
    description: Optional[str] = Field(None, description="节点描述")
    status: Optional[ComputeNodeStatus] = Field(None, description="节点状态")
    type: Optional[ComputeNodeType] = Field(None, description="节点类型")
    ip_address: Optional[IPvAnyAddress] = Field(None, description="节点IP地址")
    port: Optional[conint(gt=0, lt=65536)] = Field(None, description="节点端口")
    resources: Optional[Dict] = Field(None, description="节点资源信息")


class ComputeNodeResponse(ComputeNodeBase):
    """计算节点响应模型"""

    id: UUID = Field(..., description="计算节点ID")
    status: ComputeNodeStatus = Field(..., description="节点状态")
    created_at: Optional[datetime] = Field(..., description="创建时间")
    updated_at: Optional[datetime] = Field(..., description="更新时间")

    model_config = ConfigDict(from_attributes=True)


class ComputeNodeList(ListResponse[ComputeNodeResponse]):
    """计算节点列表响应模型"""

    pass
