from tortoise import fields
from tortoise.models import Model

from algo_flow.cores.constant.algo import (
    ComputeNodeStatus,
    ComputeNodeType,
    DatasetStatus,
    ModelFormat,
    ModelStatus,
    ModelType,
    PipelineStepType,
    ProjectStatus,
    TaskStatus,
)


class Project(Model):
    """项目模型

    用于管理机器学习项目，包含项目的基本信息和状态。
    一个项目可以包含多个模型，用于组织和管理相关的模型资源。
    """

    id = fields.UUIDField(primary_key=True, description="项目唯一标识符")
    name = fields.CharField(max_length=100, description="项目名称")
    description = fields.TextField(null=True, description="项目描述")
    status = fields.IntEnumField(
        ProjectStatus, default=ProjectStatus.ACTIVE, description="项目状态"
    )
    created_at = fields.DatetimeField(auto_now_add=True, description="创建时间")
    updated_at = fields.DatetimeField(auto_now=True, description="更新时间")

    class Meta:
        table = "algo_projects"
        ordering = ["-updated_at"]


class ModelInfo(Model):
    """模型信息模型

    用于存储机器学习模型的基本信息，包括模型类型、格式、状态等。
    每个模型必须属于一个项目，可以包含多个训练、验证和推理任务。
    """

    id = fields.UUIDField(primary_key=True, description="模型唯一标识符")
    project = fields.ForeignKeyField(
        "models.Project", related_name="models", description="所属项目"
    )
    name = fields.CharField(max_length=100, description="模型名称")
    description = fields.TextField(null=True, description="模型描述")
    version = fields.CharField(max_length=50, description="模型版本号")
    type = fields.IntEnumField(ModelType, description="模型类型")
    format = fields.IntEnumField(ModelFormat, description="模型格式")
    status = fields.IntEnumField(ModelStatus, default=ModelStatus.DRAFT, description="模型状态")
    storage_path = fields.CharField(max_length=255, null=True, description="模型文件存储路径")
    config = fields.JSONField(description="模型配置信息")
    metrics = fields.JSONField(null=True, description="模型评估指标")
    created_at = fields.DatetimeField(auto_now_add=True, description="创建时间")
    updated_at = fields.DatetimeField(auto_now=True, description="更新时间")

    class Meta:
        table = "algo_models"
        ordering = ["-updated_at"]


class Dataset(Model):
    """数据集模型

    用于管理机器学习数据集，包含数据集的基本信息和状态。
    数据集是独立的资源，可以在多个训练和验证任务中重复使用。
    """

    id = fields.UUIDField(primary_key=True, description="数据集唯一标识符")
    name = fields.CharField(max_length=100, description="数据集名称")
    description = fields.TextField(null=True, description="数据集描述")
    status = fields.IntEnumField(
        DatasetStatus, default=DatasetStatus.PROCESSING, description="数据集状态"
    )
    storage_path = fields.CharField(max_length=255, description="数据集文件存储路径")
    metadata = fields.JSONField(description="数据集元数据信息")
    created_at = fields.DatetimeField(auto_now_add=True, description="创建时间")
    updated_at = fields.DatetimeField(auto_now=True, description="更新时间")

    class Meta:
        table = "algo_datasets"
        ordering = ["-updated_at"]


class ComputeNode(Model):
    """计算节点模型

    用于管理计算资源节点，记录节点的基本信息和状态。
    用于任务调度和资源分配。
    """

    id = fields.UUIDField(primary_key=True, description="计算节点唯一标识符")
    name = fields.CharField(max_length=100, description="节点名称")
    description = fields.TextField(null=True, description="节点描述")
    status = fields.IntEnumField(
        ComputeNodeStatus, default=ComputeNodeStatus.OFFLINE, description="节点状态"
    )
    type = fields.IntEnumField(ComputeNodeType, description="节点类型")
    ip_address = fields.CharField(max_length=50, description="节点IP地址")
    port = fields.IntField(description="节点端口")
    resources = fields.JSONField(description="节点资源信息")
    created_at = fields.DatetimeField(auto_now_add=True, description="创建时间")
    updated_at = fields.DatetimeField(auto_now=True, description="更新时间")

    class Meta:
        table = "algo_compute_nodes"
        ordering = ["-updated_at"]


class Pipeline(Model):
    """管道模型

    用于定义和管理机器学习工作流中的数据处理和模型训练流程。
    管道可以包含多个步骤，每个步骤可以是数据预处理、模型训练、验证等任务。
    """

    id = fields.UUIDField(primary_key=True, description="管道唯一标识符")
    name = fields.CharField(max_length=100, description="管道名称")
    description = fields.TextField(null=True, description="管道描述")
    status = fields.IntEnumField(TaskStatus, default=TaskStatus.PENDING, description="管道状态")
    steps = fields.JSONField(description="管道步骤配置")
    created_at = fields.DatetimeField(auto_now_add=True, description="创建时间")
    updated_at = fields.DatetimeField(auto_now=True, description="更新时间")

    class Meta:
        table = "algo_pipelines"
        ordering = ["-updated_at"]


class PipelineStep(Model):
    """管道步骤模型

    用于定义管道中的单个步骤，记录步骤的类型、配置和状态。
    每个步骤可以是数据处理、模型训练或验证等任务。
    """

    id = fields.UUIDField(primary_key=True, description="步骤唯一标识符")
    name = fields.CharField(max_length=100, description="步骤名称")
    type = fields.IntEnumField(PipelineStepType, description="步骤类型")
    config = fields.JSONField(description="步骤配置")
    status = fields.IntEnumField(TaskStatus, default=TaskStatus.PENDING, description="步骤状态")
    created_at = fields.DatetimeField(auto_now_add=True, description="创建时间")
    updated_at = fields.DatetimeField(auto_now=True, description="更新时间")

    class Meta:
        table = "algo_pipeline_steps"
        ordering = ["-updated_at"]
