from typing import List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.algo.models import Project
from algo_flow.cores.constant.algo import ProjectStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_project(
    name: str,
    description: Optional[str] = None,
    status: ProjectStatus = ProjectStatus.ACTIVE,
) -> Project:
    """
    创建新项目

    Args:
        name: 项目名称
        description: 项目描述
        status: 项目状态

    Returns:
        Project: 创建的项目实例

    Raises:
        ResourceConflictError: 当项目名称已存在时抛出
    """
    # 检查项目名称是否已存在
    existing = await Project.filter(name=name).first()
    if existing:
        raise ResourceConflictError(f"项目名称 '{name}' 已存在")

    # 创建新项目
    project = await Project.create(name=name, description=description, status=status)

    return project


async def get_project(project_id: UUID) -> Project:
    """
    获取项目详情

    Args:
        project_id: 项目ID

    Returns:
        Project: 项目实例

    Raises:
        ResourceNotFoundError: 当项目不存在时抛出
    """
    try:
        project = await Project.get(id=project_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"项目 {project_id} 不存在")

    return project


async def list_projects(
    status: Optional[ProjectStatus] = None, offset: int = 0, limit: int = 10
) -> List[Project]:
    """
    获取项目列表

    Args:
        status: 可选的项目状态过滤
        offset: 分页偏移量
        limit: 分页大小

    Returns:
        List[Project]: 项目列表
    """
    query = Project.all()

    if status is not None:
        query = query.filter(status=status)

    projects = await query.offset(offset).limit(limit)
    return projects


async def update_project(
    project_id: UUID,
    name: Optional[str] = None,
    description: Optional[str] = None,
    status: Optional[ProjectStatus] = None,
) -> Project:
    """
    更新项目信息

    Args:
        project_id: 项目ID
        name: 新的项目名称
        description: 新的项目描述
        status: 新的项目状态

    Returns:
        Project: 更新后的项目实例

    Raises:
        ResourceNotFoundError: 当项目不存在时抛出
        ResourceConflictError: 当新项目名称已存在时抛出
    """
    try:
        project = await Project.get(id=project_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"项目 {project_id} 不存在")

    if name is not None and name != project.name:
        # 检查新名称是否已存在
        existing = await Project.filter(name=name).first()
        if existing:
            raise ResourceConflictError(f"项目名称 '{name}' 已存在")
        project.name = name

    if description is not None:
        project.description = description

    if status is not None:
        project.status = status

    await project.save()
    return project


@atomic()
async def delete_project(project_id: UUID) -> None:
    """
    删除项目

    Args:
        project_id: 项目ID

    Raises:
        ResourceNotFoundError: 当项目不存在时抛出
    """
    try:
        project = await Project.get(id=project_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"项目 {project_id} 不存在")

    # 删除项目（由于设置了级联删除，相关的模型等资源会自动删除）
    await project.delete()


async def count_projects(status: Optional[ProjectStatus] = None) -> int:
    """
    获取项目总数

    Args:
        status: 可选的项目状态过滤

    Returns:
        int: 项目总数
    """
    query = Project.all()

    if status is not None:
        query = query.filter(status=status)

    return await query.count()
