from typing import List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.algo.models import PipelineStep
from algo_flow.cores.constant.algo import PipelineStepType, TaskStatus
from algo_flow.cores.exceptions import ResourceNotFoundError


async def create_pipeline_step(
    name: str,
    type: PipelineStepType,
    config: any = None,
    status: TaskStatus = TaskStatus.PENDING,
) -> PipelineStep:
    step = await PipelineStep.create(name=name, type=type, config=config, status=status)
    return step


async def get_pipeline_step(step_id: UUID) -> PipelineStep:
    try:
        step = await PipelineStep.get(id=step_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"步骤 {step_id} 不存在")
    return step


async def list_pipeline_steps(
    type: Optional[PipelineStepType] = None,
    status: Optional[TaskStatus] = None,
    offset: int = 0,
    limit: int = 10,
) -> List[PipelineStep]:
    query = PipelineStep.all()
    if type is not None:
        query = query.filter(type=type)
    if status is not None:
        query = query.filter(status=status)
    steps = await query.offset(offset).limit(limit)
    return steps


async def update_pipeline_step(
    step_id: UUID,
    name: Optional[str] = None,
    type: Optional[PipelineStepType] = None,
    config: Optional[any] = None,
    status: Optional[TaskStatus] = None,
) -> PipelineStep:
    try:
        step = await PipelineStep.get(id=step_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"步骤 {step_id} 不存在")
    if name is not None:
        step.name = name
    if type is not None:
        step.type = type
    if config is not None:
        step.config = config
    if status is not None:
        step.status = status
    await step.save()
    return step


@atomic()
async def delete_pipeline_step(step_id: UUID) -> None:
    try:
        step = await PipelineStep.get(id=step_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"步骤 {step_id} 不存在")
    await step.delete()


async def count_pipeline_steps(
    type: Optional[PipelineStepType] = None, status: Optional[TaskStatus] = None
) -> int:
    query = PipelineStep.all()
    if type is not None:
        query = query.filter(type=type)
    if status is not None:
        query = query.filter(status=status)
    return await query.count()
