from typing import List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.algo.models import Pipeline
from algo_flow.cores.constant.algo import TaskStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_pipeline(
    name: str,
    description: Optional[str] = None,
    steps: any = None,
    status: TaskStatus = TaskStatus.PENDING,
) -> Pipeline:
    existing = await Pipeline.filter(name=name).first()
    if existing:
        raise ResourceConflictError(f"管道名称 '{name}' 已存在")
    pipeline = await Pipeline.create(name=name, description=description, steps=steps, status=status)
    return pipeline


async def get_pipeline(pipeline_id: UUID) -> Pipeline:
    try:
        pipeline = await Pipeline.get(id=pipeline_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"管道 {pipeline_id} 不存在")
    return pipeline


async def list_pipelines(
    status: Optional[TaskStatus] = None, offset: int = 0, limit: int = 10
) -> List[Pipeline]:
    query = Pipeline.all()
    if status is not None:
        query = query.filter(status=status)
    pipelines = await query.offset(offset).limit(limit)
    return pipelines


async def update_pipeline(
    pipeline_id: UUID,
    name: Optional[str] = None,
    description: Optional[str] = None,
    status: Optional[TaskStatus] = None,
    steps: Optional[any] = None,
) -> Pipeline:
    try:
        pipeline = await Pipeline.get(id=pipeline_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"管道 {pipeline_id} 不存在")
    if name is not None and name != pipeline.name:
        existing = await Pipeline.filter(name=name).first()
        if existing:
            raise ResourceConflictError(f"管道名称 '{name}' 已存在")
        pipeline.name = name
    if description is not None:
        pipeline.description = description
    if status is not None:
        pipeline.status = status
    if steps is not None:
        pipeline.steps = steps
    await pipeline.save()
    return pipeline


@atomic()
async def delete_pipeline(pipeline_id: UUID) -> None:
    try:
        pipeline = await Pipeline.get(id=pipeline_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"管道 {pipeline_id} 不存在")
    await pipeline.delete()


async def count_pipelines(status: Optional[TaskStatus] = None) -> int:
    query = Pipeline.all()
    if status is not None:
        query = query.filter(status=status)
    return await query.count()
