from typing import Dict, List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.algo.models import Dataset
from algo_flow.cores.constant.algo import DatasetStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_dataset(
    name: str,
    storage_path: str,
    metadata: Dict,
    description: Optional[str] = None,
    status: DatasetStatus = DatasetStatus.PROCESSING,
) -> Dataset:
    """
    创建新数据集

    Args:
        name: 数据集名称
        storage_path: 数据集文件存储路径
        metadata: 数据集元数据信息
        description: 数据集描述
        status: 数据集状态

    Returns:
        Dataset: 创建的数据集实例

    Raises:
        ResourceConflictError: 当数据集名称已存在时抛出
    """
    # 检查数据集名称是否已存在
    existing = await Dataset.filter(name=name).first()
    if existing:
        raise ResourceConflictError(f"数据集名称 '{name}' 已存在")

    # 创建新数据集
    dataset = await Dataset.create(
        name=name,
        description=description,
        status=status,
        storage_path=storage_path,
        metadata=metadata,
    )

    return dataset


async def get_dataset(dataset_id: UUID) -> Dataset:
    """
    获取数据集详情

    Args:
        dataset_id: 数据集ID

    Returns:
        Dataset: 数据集实例

    Raises:
        ResourceNotFoundError: 当数据集不存在时抛出
    """
    try:
        dataset = await Dataset.get(id=dataset_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"数据集 {dataset_id} 不存在")

    return dataset


async def list_datasets(
    status: Optional[DatasetStatus] = None, offset: int = 0, limit: int = 10
) -> List[Dataset]:
    """
    获取数据集列表

    Args:
        status: 可选的数据集状态过滤
        offset: 分页偏移量
        limit: 分页大小

    Returns:
        List[Dataset]: 数据集列表
    """
    query = Dataset.all()

    if status is not None:
        query = query.filter(status=status)

    datasets = await query.offset(offset).limit(limit)
    return datasets


async def update_dataset(
    dataset_id: UUID,
    name: Optional[str] = None,
    description: Optional[str] = None,
    status: Optional[DatasetStatus] = None,
    storage_path: Optional[str] = None,
    metadata: Optional[Dict] = None,
) -> Dataset:
    """
    更新数据集信息

    Args:
        dataset_id: 数据集ID
        name: 新的数据集名称
        description: 新的数据集描述
        status: 新的数据集状态
        storage_path: 新的数据集文件存储路径
        metadata: 新的数据集元数据信息

    Returns:
        Dataset: 更新后的数据集实例

    Raises:
        ResourceNotFoundError: 当数据集不存在时抛出
        ResourceConflictError: 当新数据集名称已存在时抛出
    """
    try:
        dataset = await Dataset.get(id=dataset_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"数据集 {dataset_id} 不存在")

    if name is not None and name != dataset.name:
        # 检查新名称是否已存在
        existing = await Dataset.filter(name=name).first()
        if existing:
            raise ResourceConflictError(f"数据集名称 '{name}' 已存在")
        dataset.name = name

    if description is not None:
        dataset.description = description

    if status is not None:
        dataset.status = status

    if storage_path is not None:
        dataset.storage_path = storage_path

    if metadata is not None:
        dataset.metadata = metadata

    await dataset.save()
    return dataset


@atomic()
async def delete_dataset(dataset_id: UUID) -> None:
    """
    删除数据集

    Args:
        dataset_id: 数据集ID

    Raises:
        ResourceNotFoundError: 当数据集不存在时抛出
    """
    try:
        dataset = await Dataset.get(id=dataset_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"数据集 {dataset_id} 不存在")

    # 删除数据集（由于设置了级联删除，相关的训练任务等资源会自动删除）
    await dataset.delete()


async def count_datasets(
    status: Optional[DatasetStatus] = None,
) -> int:
    """
    获取数据集总数

    Args:
        status: 可选的数据集状态过滤

    Returns:
        int: 数据集总数
    """
    query = Dataset.all()

    if status is not None:
        query = query.filter(status=status)

    return await query.count()
