from typing import Dict, List, Optional
from uuid import UUID

from tortoise.exceptions import DoesNotExist
from tortoise.transactions import atomic

from algo_flow.app.algo.models import ComputeNode
from algo_flow.cores.constant.algo import ComputeNodeStatus, ComputeNodeType
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


async def create_compute_node(
    name: str,
    type: ComputeNodeType,
    ip_address: str,
    port: int,
    resources: Dict,
    description: Optional[str] = None,
    status: ComputeNodeStatus = ComputeNodeStatus.OFFLINE,
) -> ComputeNode:
    """
    创建新计算节点

    Args:
        name: 节点名称
        type: 节点类型
        ip_address: 节点IP地址
        port: 节点端口
        resources: 节点资源信息
        description: 节点描述
        status: 节点状态

    Returns:
        ComputeNode: 创建的计算节点实例

    Raises:
        ResourceConflictError: 当节点名称已存在或IP地址和端口组合已存在时抛出
    """
    # 检查节点名称是否已存在
    existing = await ComputeNode.filter(name=name).first()
    if existing:
        raise ResourceConflictError(f"节点名称 '{name}' 已存在")

    # 检查IP地址和端口组合是否已存在
    existing = await ComputeNode.filter(ip_address=ip_address, port=port).first()
    if existing:
        raise ResourceConflictError(f"IP地址 {ip_address} 和端口 {port} 的组合已存在")

    # 创建新计算节点
    node = await ComputeNode.create(
        name=name,
        description=description,
        status=status,
        type=type,
        ip_address=ip_address,
        port=port,
        resources=resources,
    )

    return node


async def get_compute_node(node_id: UUID) -> ComputeNode:
    """
    获取计算节点详情

    Args:
        node_id: 计算节点ID

    Returns:
        ComputeNode: 计算节点实例

    Raises:
        ResourceNotFoundError: 当计算节点不存在时抛出
    """
    try:
        node = await ComputeNode.get(id=node_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"计算节点 {node_id} 不存在")

    return node


async def list_compute_nodes(
    status: Optional[ComputeNodeStatus] = None,
    type: Optional[ComputeNodeType] = None,
    offset: int = 0,
    limit: int = 10,
) -> List[ComputeNode]:
    """
    获取计算节点列表

    Args:
        status: 可选的节点状态过滤
        type: 可选的节点类型过滤
        offset: 分页偏移量
        limit: 分页大小

    Returns:
        List[ComputeNode]: 计算节点列表
    """
    query = ComputeNode.all()

    if status is not None:
        query = query.filter(status=status)

    if type is not None:
        query = query.filter(type=type)

    nodes = await query.offset(offset).limit(limit)
    return nodes


async def _validate_compute_node_name(node: ComputeNode, new_name: str) -> None:
    """验证计算节点名称是否可用"""
    if new_name != node.name:
        existing = await ComputeNode.filter(name=new_name).first()
        if existing:
            raise ResourceConflictError(f"节点名称 '{new_name}' 已存在")


async def _validate_compute_node_address(node: ComputeNode, new_ip: str, new_port: int) -> None:
    """验证计算节点地址是否可用"""
    if new_ip != node.ip_address or new_port != node.port:
        existing = await ComputeNode.filter(ip_address=new_ip, port=new_port).first()
        if existing:
            raise ResourceConflictError(f"IP地址 {new_ip} 和端口 {new_port} 的组合已存在")


async def update_compute_node(
    node_id: UUID,
    name: Optional[str] = None,
    description: Optional[str] = None,
    status: Optional[ComputeNodeStatus] = None,
    type: Optional[ComputeNodeType] = None,
    ip_address: Optional[str] = None,
    port: Optional[int] = None,
    resources: Optional[Dict] = None,
) -> ComputeNode:
    """
    更新计算节点信息

    Args:
        node_id: 计算节点ID
        name: 新的节点名称
        description: 新的节点描述
        status: 新的节点状态
        type: 新的节点类型
        ip_address: 新的节点IP地址
        port: 新的节点端口
        resources: 新的节点资源信息

    Returns:
        ComputeNode: 更新后的计算节点实例

    Raises:
        ResourceNotFoundError: 当计算节点不存在时抛出
        ResourceConflictError: 当新节点名称已存在或新IP地址和端口组合已存在时抛出
    """
    try:
        node = await ComputeNode.get(id=node_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"计算节点 {node_id} 不存在")

    # 验证名称
    if name is not None:
        await _validate_compute_node_name(node, name)
        node.name = name

    # 验证地址
    if ip_address is not None or port is not None:
        new_ip = ip_address if ip_address is not None else node.ip_address
        new_port = port if port is not None else node.port
        await _validate_compute_node_address(node, new_ip, new_port)
        node.ip_address = new_ip
        node.port = new_port

    # 更新其他字段
    if description is not None:
        node.description = description
    if status is not None:
        node.status = status
    if type is not None:
        node.type = type
    if resources is not None:
        node.resources = resources

    await node.save()
    return node


@atomic()
async def delete_compute_node(node_id: UUID) -> None:
    """
    删除计算节点

    Args:
        node_id: 计算节点ID

    Raises:
        ResourceNotFoundError: 当计算节点不存在时抛出
    """
    try:
        node = await ComputeNode.get(id=node_id)
    except DoesNotExist:
        raise ResourceNotFoundError(f"计算节点 {node_id} 不存在")

    await node.delete()


async def count_compute_nodes(
    status: Optional[ComputeNodeStatus] = None,
    type: Optional[ComputeNodeType] = None,
) -> int:
    """
    获取计算节点总数

    Args:
        status: 可选的节点状态过滤
        type: 可选的节点类型过滤

    Returns:
        int: 计算节点总数
    """
    query = ComputeNode.all()

    if status is not None:
        query = query.filter(status=status)

    if type is not None:
        query = query.filter(type=type)

    return await query.count()
