"use strict";
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const aws_prototyping_1 = require("../src/packs/aws-prototyping");
const expectedErrors = [
    "AwsPrototyping-AutoScalingLaunchConfigPublicIpDisabled",
    "AwsPrototyping-Cloud9InstanceNoIngressSystemsManager",
    "AwsPrototyping-CloudFrontDistributionGeoRestrictions",
    "AwsPrototyping-CloudFrontDistributionS3OriginAccessIdentity",
    "AwsPrototyping-CodeBuildProjectEnvVarAwsCred",
    "AwsPrototyping-CognitoUserPoolNoUnauthenticatedLogins",
    "AwsPrototyping-CognitoUserPoolStrongPasswordPolicy",
    "AwsPrototyping-DMSReplicationNotPublic",
    "AwsPrototyping-DocumentDBCredentialsInSecretsManager",
    "AwsPrototyping-EC2InstancesInVPC",
    "AwsPrototyping-EC2RestrictedCommonPorts",
    "AwsPrototyping-EC2RestrictedInbound",
    "AwsPrototyping-EC2RestrictedSSH",
    "AwsPrototyping-ECROpenAccess",
    "AwsPrototyping-EKSClusterNoEndpointPublicAccess",
    "AwsPrototyping-ElastiCacheClusterInVPC",
    "AwsPrototyping-ElasticBeanstalkManagedUpdatesEnabled",
    "AwsPrototyping-EventBusOpenAccess",
    "AwsPrototyping-IAMPolicyNoStatementsWithAdminAccess",
    "AwsPrototyping-LambdaFunctionPublicAccessProhibited",
    "AwsPrototyping-LambdaFunctionUrlAuth",
    "AwsPrototyping-NeptuneClusterAutomaticMinorVersionUpgrade",
    "AwsPrototyping-OpenSearchInVPCOnly",
    "AwsPrototyping-OpenSearchNoUnsignedOrAnonymousAccess",
    "AwsPrototyping-RDSAutomaticMinorVersionUpgradeEnabled",
    "AwsPrototyping-RDSInstancePublicAccess",
    "AwsPrototyping-RDSRestrictedInbound",
    "AwsPrototyping-RedshiftClusterInVPC",
    "AwsPrototyping-RedshiftClusterPublicAccess",
    "AwsPrototyping-RedshiftClusterVersionUpgrade",
    "AwsPrototyping-S3BucketLevelPublicAccessProhibited",
    "AwsPrototyping-S3BucketPublicReadProhibited",
    "AwsPrototyping-S3BucketPublicWriteProhibited",
    "AwsPrototyping-S3WebBucketOAIAccess",
    "AwsPrototyping-VPCDefaultSecurityGroupClosed",
    "AwsPrototyping-VPCSubnetAutoAssignPublicIpDisabled",
];
const expectedWarnings = [
    "AwsPrototyping-APIGWAssociatedWithWAF",
    "AwsPrototyping-APIGWAuthorization",
    "AwsPrototyping-APIGWCacheEnabledAndEncrypted",
    "AwsPrototyping-APIGWRequestValidation",
    "AwsPrototyping-APIGWSSLEnabled",
    "AwsPrototyping-AppSyncGraphQLRequestLogging",
    "AwsPrototyping-AthenaWorkgroupEncryptedQueryResults",
    "AwsPrototyping-AutoScalingGroupELBHealthCheckRequired",
    "AwsPrototyping-CloudFrontDistributionHttpsViewerNoOutdatedSSL",
    "AwsPrototyping-CloudFrontDistributionNoOutdatedSSL",
    "AwsPrototyping-CloudFrontDistributionWAFIntegration",
    "AwsPrototyping-CloudTrailEncryptionEnabled",
    "AwsPrototyping-CloudWatchLogGroupEncrypted",
    "AwsPrototyping-CodeBuildProjectKMSEncryptedArtifacts",
    "AwsPrototyping-CodeBuildProjectManagedImages",
    "AwsPrototyping-CodeBuildProjectPrivilegedModeDisabled",
    "AwsPrototyping-CodeBuildProjectSourceRepoUrl",
    "AwsPrototyping-CognitoUserPoolAdvancedSecurityModeEnforced",
    "AwsPrototyping-CognitoUserPoolAPIGWAuthorizer",
    "AwsPrototyping-CognitoUserPoolMFA",
    "AwsPrototyping-DocumentDBClusterEncryptionAtRest",
    "AwsPrototyping-DocumentDBClusterNonDefaultPort",
    "AwsPrototyping-DAXEncrypted",
    "AwsPrototyping-EC2EBSVolumeEncrypted",
    "AwsPrototyping-EC2InstanceNoPublicIp",
    "AwsPrototyping-EC2InstanceProfileAttached",
    "AwsPrototyping-EC2InstanceTerminationProtection",
    "AwsPrototyping-EC2SecurityGroupDescription",
    "AwsPrototyping-ECSClusterCloudWatchContainerInsights",
    "AwsPrototyping-ECSTaskDefinitionContainerLogging",
    "AwsPrototyping-ECSTaskDefinitionNoEnvironmentVariables",
    "AwsPrototyping-ECSTaskDefinitionUserForHostMode",
    "AwsPrototyping-EFSEncrypted",
    "AwsPrototyping-EKSClusterControlPlaneLogs",
    "AwsPrototyping-ElastiCacheClusterNonDefaultPort",
    "AwsPrototyping-ElastiCacheRedisClusterEncryption",
    "AwsPrototyping-ElastiCacheRedisClusterRedisAuth",
    "AwsPrototyping-ElasticBeanstalkEC2InstanceLogsToS3",
    "AwsPrototyping-ElasticBeanstalkVPCSpecified",
    "AwsPrototyping-ALBHttpDropInvalidHeaderEnabled",
    "AwsPrototyping-ALBHttpToHttpsRedirection",
    "AwsPrototyping-ALBWAFEnabled",
    "AwsPrototyping-CLBConnectionDraining",
    "AwsPrototyping-ELBACMCertificateRequired",
    "AwsPrototyping-ELBTlsHttpsListenersOnly",
    "AwsPrototyping-ELBv2ACMCertificateRequired",
    "AwsPrototyping-EMRAuthEC2KeyPairOrKerberos",
    "AwsPrototyping-EMREncryptionInTransit",
    "AwsPrototyping-EMRKerberosEnabled",
    "AwsPrototyping-EMRLocalDiskEncryption",
    "AwsPrototyping-GlueEncryptedCloudWatchLogs",
    "AwsPrototyping-GlueJobBookmarkEncrypted",
    "AwsPrototyping-IAMNoManagedPolicies",
    "AwsPrototyping-IAMNoWildcardPermissions",
    "AwsPrototyping-IAMPolicyNoStatementsWithFullAccess",
    "AwsPrototyping-KinesisDataFirehoseSSE",
    "AwsPrototyping-KinesisDataStreamDefaultKeyWhenSSE",
    "AwsPrototyping-KinesisDataStreamSSE",
    "AwsPrototyping-KMSBackingKeyRotationEnabled",
    "AwsPrototyping-LambdaInsideVPC",
    "AwsPrototyping-LambdaLatestVersion",
    "AwsPrototyping-MediaStoreCloudWatchMetricPolicy",
    "AwsPrototyping-MediaStoreContainerCORSPolicy",
    "AwsPrototyping-MediaStoreContainerHasContainerPolicy",
    "AwsPrototyping-MediaStoreContainerLifecyclePolicy",
    "AwsPrototyping-MediaStoreContainerSSLRequestsOnly",
    "AwsPrototyping-MSKBrokerToBrokerTLS",
    "AwsPrototyping-MSKClientToBrokerTLS",
    "AwsPrototyping-NeptuneClusterEncryptionAtRest",
    "AwsPrototyping-NeptuneClusterIAMAuth",
    "AwsPrototyping-OpenSearchAllowlistedIPs",
    "AwsPrototyping-OpenSearchDedicatedMasterNode",
    "AwsPrototyping-OpenSearchEncryptedAtRest",
    "AwsPrototyping-OpenSearchNodeToNodeEncryption",
    "AwsPrototyping-QuicksightSSLConnections",
    "AwsPrototyping-AuroraMySQLPostgresIAMAuth",
    "AwsPrototyping-RDSInstanceDeletionProtectionEnabled",
    "AwsPrototyping-RDSNonDefaultPort",
    "AwsPrototyping-RDSStorageEncrypted",
    "AwsPrototyping-RedshiftClusterAuditLogging",
    "AwsPrototyping-RedshiftClusterConfiguration",
    "AwsPrototyping-RedshiftClusterEncryptionAtRest",
    "AwsPrototyping-RedshiftClusterMaintenanceSettings",
    "AwsPrototyping-RedshiftClusterNonDefaultPort",
    "AwsPrototyping-RedshiftClusterNonDefaultUsername",
    "AwsPrototyping-RedshiftEnhancedVPCRoutingEnabled",
    "AwsPrototyping-RedshiftRequireTlsSSL",
    "AwsPrototyping-S3BucketLoggingEnabled",
    "AwsPrototyping-S3BucketServerSideEncryptionEnabled",
    "AwsPrototyping-S3BucketSSLRequestsOnly",
    "AwsPrototyping-S3DefaultEncryptionKMS",
    "AwsPrototyping-SageMakerEndpointConfigurationKMSKeyConfigured",
    "AwsPrototyping-SageMakerNotebookInstanceKMSKeyConfigured",
    "AwsPrototyping-SageMakerNotebookInVPC",
    "AwsPrototyping-SageMakerNotebookNoDirectInternetAccess",
    "AwsPrototyping-SecretsManagerRotationEnabled",
    "AwsPrototyping-SecretsManagerUsingKMSKey",
    "AwsPrototyping-SNSEncryptedKMS",
    "AwsPrototyping-SNSTopicSSLPublishOnly",
    "AwsPrototyping-SQSQueueDLQ",
    "AwsPrototyping-SQSQueueSSE",
    "AwsPrototyping-SQSQueueSSLRequestsOnly",
    "AwsPrototyping-TimestreamDatabaseCustomerManagedKey",
    "AwsPrototyping-VPCNoNACLs",
    "AwsPrototyping-VPCNoUnrestrictedRouteToIGW",
];
describe("Check NagPack Details", () => {
    describe("AwsPrototyping-CDK", () => {
        class AwsPrototypingChecksExtended extends aws_prototyping_1.AwsPrototypingChecks {
            constructor() {
                super(...arguments);
                this.actualWarnings = new Array();
                this.actualErrors = new Array();
            }
            applyRule(params) {
                const ruleSuffix = params.ruleSuffixOverride
                    ? params.ruleSuffixOverride
                    : params.rule.name;
                const ruleId = `${pack.readPackName}-${ruleSuffix}`;
                if (params.level === cdk_nag_1.NagMessageLevel.WARN) {
                    this.actualWarnings.push(ruleId);
                }
                else {
                    this.actualErrors.push(ruleId);
                }
            }
        }
        const pack = new AwsPrototypingChecksExtended();
        test("Pack Name is correct", () => {
            expect(pack.readPackName).toStrictEqual("AwsPrototyping");
        });
        test("Pack contains expected warning and error rules", () => {
            jest.spyOn(pack, "applyRule");
            const stack = new aws_cdk_lib_1.Stack();
            aws_cdk_lib_1.Aspects.of(stack).add(pack);
            new aws_cdk_lib_1.CfnResource(stack, "rTestResource", { type: "foo" });
            assert_1.SynthUtils.synthesize(stack).messages;
            expect(pack.actualWarnings.sort()).toEqual(expectedWarnings.sort());
            expect(pack.actualErrors.sort()).toEqual(expectedErrors.sort());
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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