"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleMetadata = exports.PackName = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const cdk_nag_1 = require("cdk-nag");
/**
 * Pack name
 *
 */
exports.PackName = "AwsPrototyping";
/**
 * Rule metadata
 *
 */
exports.RuleMetadata = [
    {
        info: "The REST API stage is not associated with AWS WAFv2 web ACL.",
        explanation: "AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAssociatedWithWAF,
    },
    {
        info: "The API does not implement authorization.",
        explanation: "In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.\n\nExample threat: An actor with a network path to an API gateway stage end-point can interact with the API method in question without authorization, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAuthorization,
    },
    {
        info: "The API Gateway stage does not does not have caching and encryption enabled for all methods.",
        explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWCacheEnabledAndEncrypted,
    },
    {
        info: "The REST API does not have request validation enabled.",
        explanation: "The API should have basic request validation enabled. If the API is integrated with a custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWRequestValidation,
    },
    {
        info: "The API Gateway REST API stage is not configured with SSL certificates.",
        explanation: "Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWSSLEnabled,
    },
    {
        info: "The GraphQL API does not have request level logging enabled.",
        explanation: "It is important to use CloudWatch Logs to log metrics such as who has accessed the GraphQL API, how the caller accessed the API, and invalid requests.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.appsync.AppSyncGraphQLRequestLogging,
    },
    {
        info: "The Athena workgroup does not encrypt query results.",
        explanation: "Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.athena.AthenaWorkgroupEncryptedQueryResults,
    },
    {
        info: "The Auto Scaling group (which is associated with a load balancer) does not utilize ELB health checks.",
        explanation: "The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.autoscaling.AutoScalingGroupELBHealthCheckRequired,
    },
    {
        info: "The Auto Scaling launch configuration does not have public IP addresses disabled.",
        explanation: "If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.autoscaling.AutoScalingLaunchConfigPublicIpDisabled,
    },
    {
        info: "The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.",
        explanation: "SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.\n\nExample threat: A global internet-based actor can discover Cloud9 EC2 instances that have public IP addresses and that are exposing SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloud9.Cloud9InstanceNoIngressSystemsManager,
    },
    {
        info: "The CloudFront distribution may require Geo restrictions.",
        explanation: "Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.\n\nExample threat: A global internet-based actor can discover prototype web assets that are exposed via CloudFront distributions, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionGeoRestrictions,
    },
    {
        info: "The CloudFront distribution allows for SSLv3 or TLSv1 for HTTPS viewer connections.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Help protect viewer connections by specifying a viewer certificate that enforces a minimum of TLSv1.1 or TLSv1.2 in the security policy. Distributions that use that use the default CloudFront viewer certificate or use 'vip' for the SslSupportMethod are non-compliant with this rule, as the minimum security policy is set to TLSv1 regardless of the specified MinimumProtocolVersion",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionHttpsViewerNoOutdatedSSL,
    },
    {
        info: "The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionNoOutdatedSSL,
    },
    {
        info: "The CloudFront distribution does not use an origin access identity an S3 origin.",
        explanation: "Origin access identities help with security by restricting any direct access to objects through S3 URLs.\n\nExample threat: A global internet-based actor can bypass the CloudFront distribution and associated controls (e.g. geo blocking), which may lead to direct access to static assets hosted on the S3 origin possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 origin for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionS3OriginAccessIdentity,
    },
    {
        info: "The CloudFront distribution may require integration with AWS WAF.",
        explanation: "The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionWAFIntegration,
    },
    {
        info: "The trail does not have encryption enabled.",
        explanation: "Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudtrail.CloudTrailEncryptionEnabled,
    },
    {
        info: "The CloudWatch Log Group is not encrypted with an AWS KMS key.",
        explanation: "To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudwatch.CloudWatchLogGroupEncrypted,
    },
    {
        info: "The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables.",
        explanation: "Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.\n\nExample threat: An actor who can view the CodeBuild environment variables can obtain the AWS Access Key and Secret Access Key, which may lead to the actor being able to do anything the AWS keys are authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectEnvVarAwsCred,
    },
    {
        info: "The CodeBuild project does not use an AWS KMS key for encryption.",
        explanation: "Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectKMSEncryptedArtifacts,
    },
    {
        info: "The CodeBuild project does not use images provided by the CodeBuild service or have a cdk-nag suppression rule explaining the need for a custom image.",
        explanation: "Explaining differences/edits to Docker images helps operators better understand system dependencies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectManagedImages,
    },
    {
        info: "The CodeBuild project has privileged mode enabled.",
        explanation: "Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectPrivilegedModeDisabled,
    },
    {
        info: "The CodeBuild project which utilizes either a GitHub or BitBucket repository does not utilize OAUTH.",
        explanation: "OAUTH is the most secure method of authenticating your Codebuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectSourceRepoUrl,
    },
    {
        info: "The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.",
        explanation: "Advanced security features enable the system to detect and act upon malicious sign-in attempts.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAdvancedSecurityModeEnforced,
    },
    {
        info: "The API Gateway method does not use a Cognito user pool authorizer.",
        explanation: "API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAPIGWAuthorizer,
    },
    {
        info: "The Cognito user pool is not configured to require MFA.",
        explanation: "Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolMFA,
    },
    {
        info: "The Cognito identity pool allows for unauthenticated logins and does not have a cdk-nag rule suppression with a reason.",
        explanation: "Applications do not warrant unauthenticated guest access in many cases. Metadata explaining the use case allows for transparency to operators.\n\nExample threat: A global internet-based actor who has discovered a prototype endpoint with Cognito unauthenticated logins can does not need to provide credentials to interact with the endpoint, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolNoUnauthenticatedLogins,
    },
    {
        info: "The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.",
        explanation: "Strong password policies increase system security by encouraging users to create reliable and secure passwords.\n\nExample threat: An actor who has discovered a prototype endpoint with Cognito authenticated logins can perform a dictionary or brute force attack to authenticate as an authorized Cognito user, which may lead to the actor being able to do anything the associated Cognito user is authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolStrongPasswordPolicy,
    },
    {
        info: "The DMS replication instance is public.",
        explanation: "DMS replication instances can contain sensitive information and access control is required for such accounts.\n\nExample threat: A global internet-based actor can discover DMS instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.dms.DMSReplicationNotPublic,
    },
    {
        info: "The Document DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterEncryptionAtRest,
    },
    {
        info: "The Document DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterNonDefaultPort,
    },
    {
        info: "The Document DB cluster does not have the username and password stored in Secrets Manager.",
        explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.\n\nExample threat: An actor who can view the DocumentDB configuration can obtain the username and password for the DocumentDB cluster, which may lead to the actor being able to access anything the associated DocumentDB user is authorised to do possibly impacting the confidentiality, integrity and availability of the data assets hosted on the DocumentDB cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.documentdb.DocumentDBCredentialsInSecretsManager,
    },
    {
        info: "The DAX cluster does not have server-side encryption enabled.",
        explanation: "Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.dynamodb.DAXEncrypted,
    },
    {
        info: "The EBS volume has encryption disabled.",
        explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2EBSVolumeEncrypted,
    },
    {
        info: "The EC2 instance is associated with a public IP address.",
        explanation: "Amazon EC2 instances can contain sensitive information and access control is required for such resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceNoPublicIp,
    },
    {
        info: "The EC2 instance does not have an instance profile attached.",
        explanation: "EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceProfileAttached,
    },
    {
        info: "The EC2 instance is not within a VPC.",
        explanation: "Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2InstancesInVPC,
    },
    {
        info: "The EC2 instance is not part of an ASG and has Termination Protection disabled.",
        explanation: "Termination Protection safety feature should be enabled in order to protect the instances from being accidentally terminated.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceTerminationProtection,
    },
    {
        info: "The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333).",
        explanation: "Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.\n\nExample threat: A global internet-based actor can discover exposed services (e.g. Telnet, SSH, RDS, MySQL) using their common port numbers, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedCommonPorts,
    },
    {
        info: "The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.",
        explanation: "Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public EC2 instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedInbound,
    },
    {
        info: "The Security Group allows unrestricted SSH access.",
        explanation: "Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address to SSH or move laterally to non-public EC2 instances via SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedSSH,
    },
    {
        info: "The Security Group does not have a description.",
        explanation: "Descriptions help simplify operations and remove any opportunities for operator errors.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2SecurityGroupDescription,
    },
    {
        info: "The ECR Repository allows open access.",
        explanation: "Removing * principals in an ECR Repository helps protect against unauthorized access.\n\nExample threat: A global internet-based actor who has discovered a ECR repository can access the container images hosted within the repository, which may lead to information disclosure that aids in the intrusion activities being successful against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ecr.ECROpenAccess,
    },
    {
        info: "The ECS Cluster has CloudWatch Container Insights disabled.",
        explanation: "CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSClusterCloudWatchContainerInsights,
    },
    {
        info: "One or more containers in the ECS Task Definition do not have container logging enabled.",
        explanation: "Container logging allows operators to view and aggregate the logs from the container. Containers should use the 'awslogs' driver at a minimum.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionContainerLogging,
    },
    {
        info: "The ECS Task Definition includes a container definition that directly specifies environment variables.",
        explanation: "Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionNoEnvironmentVariables,
    },
    {
        info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty.",
        explanation: "If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionUserForHostMode,
    },
    {
        info: "The EFS does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.efs.EFSEncrypted,
    },
    {
        info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
        explanation: "EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format LogExport::<log> for exported logs. Example: appliesTo: ['LogExport::authenticate'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.eks.EKSClusterControlPlaneLogs,
    },
    {
        info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
        explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).\n\nExample threat: A global internet-based actor who has discovered a EKS cluster Kubernetes API server endpoint can perform reconnaissance and intrusion activities against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eks.EKSClusterNoEndpointPublicAccess,
    },
    {
        info: "The ElastiCache cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover the ElastiCache cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterInVPC,
    },
    {
        info: "The ElastiCache cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterNonDefaultPort,
    },
    {
        info: "The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.",
        explanation: "Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterEncryption,
    },
    {
        info: "The ElastiCache Redis cluster does not use Redis AUTH for user authentication.",
        explanation: "Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterRedisAuth,
    },
    {
        info: "The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.",
        explanation: "Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkEC2InstanceLogsToS3,
    },
    {
        info: "The Elastic Beanstalk environment does not have managed updates enabled.",
        explanation: "Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.\n\nExample threat: An actor with a network path to the Elastic Beanstalk environment can attempt to take advantage of a known vulnerability in a platform component used by Elastic Beanstalk, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkManagedUpdatesEnabled,
    },
    {
        info: "The Elastic Beanstalk environment is not configured to use a specific VPC.",
        explanation: "Use a non-default VPC in order to separate your environment from default resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkVPCSpecified,
    },
    {
        info: "The ALB does not have invalid HTTP header dropping enabled.",
        explanation: "Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist in these headers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ALBHttpDropInvalidHeaderEnabled,
    },
    {
        info: "The ALB's HTTP listeners are not configured to redirect to HTTPS.",
        explanation: "To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ALBHttpToHttpsRedirection,
    },
    {
        info: "The ALB is not associated with AWS WAFv2 web ACL.",
        explanation: "A WAF helps to protect your web applications or APIs against common web exploits. These web exploits may affect availability, compromise security, or consume excessive resources within your environment.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ALBWAFEnabled,
    },
    {
        info: "The CLB does not have connection draining enabled.",
        explanation: "With Connection Draining feature enabled, if an EC2 backend instance fails health checks the CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.CLBConnectionDraining,
    },
    {
        info: "The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).",
        explanation: "Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ELBACMCertificateRequired,
    },
    {
        info: "The CLB does not restrict its listeners to only the SSL and HTTPS protocols.",
        explanation: "Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ELBTlsHttpsListenersOnly,
    },
    {
        info: "The ALB, NLB, or GLB listener do not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).",
        explanation: "Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ELBv2ACMCertificateRequired,
    },
    {
        info: "The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.",
        explanation: "SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRAuthEC2KeyPairOrKerberos,
    },
    {
        info: "The EMR cluster does not use a security configuration with encryption in transit enabled and configured.",
        explanation: "EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMREncryptionInTransit,
    },
    {
        info: "The EMR cluster does not have Kerberos enabled.",
        explanation: "The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRKerberosEnabled,
    },
    {
        info: "The EMR cluster does not use a security configuration with local disk encryption enabled.",
        explanation: "Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRLocalDiskEncryption,
    },
    {
        info: "The event bus policy allows for open access.",
        explanation: 'An open policy ("*" principal without a condition) grants anonymous access to an event bus. Use a condition to limit the permission to accounts that fulfill a certain requirement, such as being a member of a certain AWS organization.\n\nExample threat: A global internet-based actor who has discovered the Event Bridge event bus (e.g. Endpoint ID) can put arbitrary events onto the bus, which may lead to which could be processed by the prototype possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype',
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eventbridge.EventBusOpenAccess,
    },
    {
        info: "The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.",
        explanation: "Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueEncryptedCloudWatchLogs,
    },
    {
        info: "The Glue job does not have use a security configuration with job bookmark encryption enabled.",
        explanation: "Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueJobBookmarkEncrypted,
    },
    {
        info: "The IAM user, role, or group uses AWS managed policies.",
        explanation: "An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Policy::<policy> for AWS managed policies. Example: appliesTo: ['Policy::arn:<AWS::Partition>:iam::aws:policy/foo'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoManagedPolicies,
    },
    {
        info: "The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.",
        explanation: "Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Action::<action> for policy actions and Resource::<resource> for resources. Example: appliesTo: ['Action::s3:*'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoWildcardPermissions,
    },
    {
        info: "The IAM policy grants admin access - meaning the policy allows a principal to perform unlimited actions on any service",
        explanation: "AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.\n\nExample threat: A global internet-based actor who has successfully obtained valid keys or a session associated of the IAM Principal associated with the IAM policy can perform unlimited AWS actions on any AWS service which are exposed via the AWS API/Management Console/CLI, which may lead to broad and unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithAdminAccess,
    },
    {
        info: "The IAM policy grants full access - meaning the policy allows unlimited actions for a given AWS service",
        explanation: "Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithFullAccess,
    },
    {
        info: "The Kinesis Data Firehose delivery stream does not have server-side encryption enabled.",
        explanation: "Enabling encryption allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataFirehoseSSE,
    },
    {
        info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
        explanation: "Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamDefaultKeyWhenSSE,
    },
    {
        info: "The Kinesis Data Stream does not have server-side encryption enabled.",
        explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamSSE,
    },
    {
        info: "The KMS Symmetric key does not have automatic key rotation enabled.",
        explanation: "KMS key rotation allow a system to set a rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kms.KMSBackingKeyRotationEnabled,
    },
    {
        info: "The Lambda function permission grants public access",
        explanation: "Public access allows anyone on the internet to perform unauthenticated actions on your function and can potentially lead to degraded availability.\n\nExample threat: A global internet-based actor who has discovered the Lambda function name or ARN can invoke/delete/modify the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionPublicAccessProhibited,
    },
    {
        info: "The Lambda Function URL allows for public, unauthenticated access.",
        explanation: "AWS Lambda Function URLs allow you to invoke your function via a HTTPS end-point, setting the authentication to NONE allows anyone on the internet to invoke your function.\n\nExample threat: A global internet-based actor who has discovered the Lambda Function URL can invoke the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionUrlAuth,
    },
    {
        info: "The Lambda function is not VPC enabled.",
        explanation: "Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.lambda.LambdaInsideVPC,
    },
    {
        info: "The non-container Lambda function is not configured to use the latest runtime version.",
        explanation: "Use the latest available runtime for the targeted language to avoid technical debt. Runtimes specific to a language or framework version are deprecated when the version reaches end of life. This rule only applies to non-container Lambda functions.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.lambda.LambdaLatestVersion,
    },
    {
        info: "The MediaStore container does not define a metric policy to send metrics to CloudWatch.",
        explanation: "Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreCloudWatchMetricPolicy,
    },
    {
        info: "The MediaStore container does not define a CORS policy.",
        explanation: "Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerCORSPolicy,
    },
    {
        info: "The MediaStore container does not define a container policy.",
        explanation: "Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerHasContainerPolicy,
    },
    {
        info: "The MediaStore container does not define a lifecycle policy.",
        explanation: "Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerLifecyclePolicy,
    },
    {
        info: "The MediaStore container does not require requests to use SSL.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerSSLRequestsOnly,
    },
    {
        info: "The MSK cluster uses plaintext communication between brokers.",
        explanation: "TLS communication secures data-in-transit by encrypting the connection between brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKBrokerToBrokerTLS,
    },
    {
        info: "The MSK cluster uses plaintext communication between clients and brokers.",
        explanation: "TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKClientToBrokerTLS,
    },
    {
        info: "The Neptune DB instance does have Auto Minor Version Upgrade enabled.",
        explanation: "The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.\n\nExample threat: An actor with a network path to the Neptune cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by Neptune, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterAutomaticMinorVersionUpgrade,
    },
    {
        info: "The Neptune DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterEncryptionAtRest,
    },
    {
        info: "The Neptune DB cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterIAMAuth,
    },
    {
        info: "The OpenSearch Service domain does not only grant access via allowlisted IP addresses.",
        explanation: "Using allowlisted IP addresses helps protect the domain against unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchAllowlistedIPs,
    },
    {
        info: "The OpenSearch Service domain does not use dedicated master nodes.",
        explanation: "Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchDedicatedMasterNode,
    },
    {
        info: "The OpenSearch Service domain does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchEncryptedAtRest,
    },
    {
        info: "The OpenSearch Service domain is not provisioned inside a VPC.",
        explanation: "Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.\n\nExample threat: A global internet-based actor can discover the OpenSearch Service domain that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchInVPCOnly,
    },
    {
        info: "The OpenSearch Service domain does not have node-to-node encryption enabled.",
        explanation: "Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNodeToNodeEncryption,
    },
    {
        info: "The OpenSearch Service domain does not allow for unsigned requests or anonymous access.",
        explanation: "Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.\n\nExample threat: An actor with a network path to the OpenSearch Service domain can directly access the domain without authentication, which may lead to allowing access to data hosted within the domain possibly impacting the confidentiality, integrity and availability of the data assets hosted on the OpenSearch Service domain for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNoUnsignedOrAnonymousAccess,
    },
    {
        info: "The Quicksight data sources connection is not configured to use SSL.",
        explanation: "SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.quicksight.QuicksightSSLConnections,
    },
    {
        info: "The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.AuroraMySQLPostgresIAMAuth,
    },
    {
        info: "The RDS DB instance does not have automatic minor version upgrades enabled.",
        explanation: "Enable automatic minor version upgrades on your Amazon Relational Database Service (RDS) instances to ensure the latest minor version updates to the Relational Database Management System (RDBMS) are installed, which may include security patches and bug fixes.\n\nExample threat: An actor with a network path to the RDS cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by RDS, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSAutomaticMinorVersionUpgradeEnabled,
    },
    {
        info: "The RDS DB instance or Aurora DB cluster does not have deletion protection enabled.",
        explanation: "Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSInstanceDeletionProtectionEnabled,
    },
    {
        info: "The RDS DB instance allows public access.",
        explanation: "Amazon RDS database instances can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the RDS DB instance endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSInstancePublicAccess,
    },
    {
        info: "The RDS instance or Aurora DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSNonDefaultPort,
    },
    {
        info: "The RDS DB Security Group allows for 0.0.0.0/0 inbound access.",
        explanation: "RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.\n\nExample threat: A global internet-based actor can discover RDS DB instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public RDS DB instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSRestrictedInbound,
    },
    {
        info: "The RDS DB instance or Aurora DB cluster does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist at rest in Amazon RDS DB instances, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSStorageEncrypted,
    },
    {
        info: "The Redshift cluster does not have audit logging enabled.",
        explanation: "Audit logging helps operators troubleshoot issues and ensure security.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterAuditLogging,
    },
    {
        info: "The Redshift cluster does not have encryption or audit logging enabled.",
        explanation: "To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterConfiguration,
    },
    {
        info: "The Redshift cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterEncryptionAtRest,
    },
    {
        info: "The Redshift cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover a RedShift cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterInVPC,
    },
    {
        info: "The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured.",
        explanation: "Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterMaintenanceSettings,
    },
    {
        info: "The Redshift cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultPort,
    },
    {
        info: 'The Redshift cluster uses the default "awsuser" username.',
        explanation: 'Using a custom user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultUsername,
    },
    {
        info: "The Redshift cluster allows public access.",
        explanation: "Amazon Redshift clusters can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the Redshift cluster endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterPublicAccess,
    },
    {
        info: "The Redshift cluster does not have version upgrade enabled.",
        explanation: "Version Upgrade must enabled on the cluster in order to automatically receive upgrades during the maintenance window.\n\nExample threat: An actor with a network path to the Redshift cluster can attempt to take advantage of a known vulnerability in a component exposed by Redshift, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterVersionUpgrade,
    },
    {
        info: "The Redshift cluster does not have enhanced VPC routing enabled.",
        explanation: "Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftEnhancedVPCRoutingEnabled,
    },
    {
        info: "The Redshift cluster does not require TLS/SSL encryption.",
        explanation: "Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftRequireTlsSSL,
    },
    {
        info: "The S3 bucket does not prohibit public access through bucket level settings.",
        explanation: "Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read or write can read or write data to or from the S3 bucket, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketLevelPublicAccessProhibited,
    },
    {
        info: "The S3 Bucket does not have server access logs enabled.",
        explanation: "Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketLoggingEnabled,
    },
    {
        info: "The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read can read data from the S3 bucket, which may lead to possibly impacting the confidentiality of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicReadProhibited,
    },
    {
        info: "The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public write can write data to, or overwrite data within the S3 bucket, which may lead to possibly impacting the integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicWriteProhibited,
    },
    {
        info: "The S3 Bucket does not have default server-side encryption enabled.",
        explanation: "Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketServerSideEncryptionEnabled,
    },
    {
        info: "The S3 Bucket or bucket policy does not require requests to use SSL/TLS.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketSSLRequestsOnly,
    },
    {
        info: "The S3 Bucket is not encrypted with a KMS Key by default.",
        explanation: "Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3DefaultEncryptionKMS,
    },
    {
        info: "The S3 static website bucket either has an open world bucket policy or does not use a CloudFront Origin Access Identity (OAI) in the bucket policy for limited getObject and/or putObject permissions.",
        explanation: "An OAI allows you to provide access to content in your S3 static website bucket through CloudFront URLs without enabling public access through an open bucket policy, disabling S3 Block Public Access settings, and/or through object ACLs.\n\nExample threat: A global internet-based actor who has discovered a S3 hosted website can discover prototype web assets that are hosted on the website, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3WebBucketOAIAccess,
    },
    {
        info: "The SageMaker resource endpoint is not encrypted with a KMS key.",
        explanation: "Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerEndpointConfigurationKMSKeyConfigured,
    },
    {
        info: "The SageMaker notebook instance does not have an encrypted storage volume.",
        explanation: "Encrypting storage volumes helps protect SageMaker data-at-rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInstanceKMSKeyConfigured,
    },
    {
        info: "The SageMaker notebook instance is not provisioned inside a VPC.",
        explanation: "Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInVPC,
    },
    {
        info: "The SageMaker notebook does not disable direct internet access.",
        explanation: "By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookNoDirectInternetAccess,
    },
    {
        info: "The secret does not have automatic rotation scheduled.",
        explanation: "Rotating secrets on a regular schedule can shorten the period a secret is active, and potentially reduce the business impact if the secret is compromised.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.secretsmanager.SecretsManagerRotationEnabled,
    },
    {
        info: "The secret is not encrypted with a KMS Customer managed key.",
        explanation: "To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.secretsmanager.SecretsManagerUsingKMSKey,
    },
    {
        info: "The SNS topic does not have KMS encryption enabled.",
        explanation: "To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS). Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSEncryptedKMS,
    },
    {
        info: "The SNS Topic does not require publishers to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition and the 'sns:Publish' action in the topic policy to force publishers to use SSL. If SSE is already enabled then this control is auto enforced.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSTopicSSLPublishOnly,
    },
    {
        info: "The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk-nag rule suppression indicating it is a DLQ.",
        explanation: "Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueDLQ,
    },
    {
        info: "The SQS Queue does not have server-side encryption enabled.",
        explanation: "Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSE,
    },
    {
        info: "The SQS queue does not require requests to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSLRequestsOnly,
    },
    {
        info: "The Timestream database does not use a Customer Managed KMS Key for at rest encryption.",
        explanation: "All Timestream tables in a database are encrypted at rest by default using an AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an periodic basis.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.timestream.TimestreamDatabaseCustomerManagedKey,
    },
    {
        info: "The VPC's default security group allows inbound or outbound traffic.",
        explanation: "When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.\n\nExample threat: A global internet-based actor can discover VPC-attached resources (e.g. EC2 instances) that have public IP addresses and allow ingress to all internet address or move laterally to non-public VPC-attached instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.vpc.VPCDefaultSecurityGroupClosed,
    },
    {
        info: "A Network ACL or Network ACL entry has been implemented.",
        explanation: "Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.vpc.VPCNoNACLs,
    },
    {
        info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0').",
        explanation: "Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.vpc.VPCNoUnrestrictedRouteToIGW,
    },
    {
        info: "The subnet auto-assigns public IP addresses.",
        explanation: "Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.\n\nExample threat: A global internet-based actor can discover VPC-attached resources (e.g. EC2 instances) within the subnet in question that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.vpc.VPCSubnetAutoAssignPublicIpDisabled,
    },
];
//# sourceMappingURL=data:application/json;base64,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