"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const synth_1 = require("projen/lib/util/synth");
const src_1 = require("../src");
describe("NX Monorepo Unit Tests", () => {
    it("Empty Monorepo", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Defaults",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Ignore Patterns", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "IgnorePatterns",
            nxConfig: {
                nxIgnore: ["pattern1.txt", "**/foo.ts"],
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Target Dependencies", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "TargetDependencies",
            nxConfig: {
                targetDependencies: {
                    test: [
                        { target: "test", projects: src_1.TargetDependencyProject.DEPENDENCIES },
                    ],
                    eslint: [
                        { target: "eslint", projects: src_1.TargetDependencyProject.SELF },
                    ],
                },
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Affected Branch", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "main",
            name: "AffectedBranch",
            nxConfig: {
                affectedBranch: "main",
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it.each([javascript_1.NodePackageManager.PNPM, javascript_1.NodePackageManager.YARN])("Additional Workspace Packages", (packageManager) => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            packageManager,
            name: "AdditionalWorkspacePackages",
            workspaceConfig: {
                additionalPackages: ["my/custom/package"],
            },
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("another/custom/package", "yet/another/package");
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Workspace Package Order", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "WorkspacePackageOrder",
            workspaceConfig: {
                additionalPackages: ["packages/one"],
            },
        });
        new typescript_1.TypeScriptProject({
            name: "two",
            outdir: "packages/two",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("packages/three", "packages/four");
        new typescript_1.TypeScriptProject({
            name: "five",
            outdir: "packages/five",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("packages/six");
        new typescript_1.TypeScriptProject({
            name: "seven",
            outdir: "packages/seven",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("PNPM", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager: javascript_1.NodePackageManager.PNPM,
            defaultReleaseBranch: "mainline",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Validate consistent Package Managers", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        expect(() => synth_1.synthSnapshot(project)).toThrow("ts-subproject packageManager does not match the monorepo packageManager: pnpm.");
    });
    it("Composite", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Composite",
        });
        const tsProject = new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject2",
            outdir: "packages/ts-subproject2",
            parent: project,
            defaultReleaseBranch: "mainline",
            devDeps: ["ts-subproject"],
        });
        const pyProject = new python_1.PythonProject({
            parent: project,
            outdir: "packages/py-subproject",
            authorEmail: "test@test.com",
            authorName: "test",
            moduleName: "py_subproject",
            name: "py-subproject",
            version: "0.0.0",
        });
        project.addImplicitDependency(tsProject, pyProject);
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
});
//# sourceMappingURL=data:application/json;base64,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