#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
快速重置熔断器脚本

当API认证问题解决后，使用此脚本重置熔断器状态。
"""

import sys
from pathlib import Path

# 修复Windows控制台编码问题
if sys.platform == 'win32':
    import ctypes
    try:
        ctypes.windll.kernel32.SetConsoleOutputCP(65001)
        ctypes.windll.kernel32.SetConsoleCP(65001)
    except Exception:
        pass
    if sys.stdout.encoding != 'utf-8':
        try:
            sys.stdout.reconfigure(encoding='utf-8')
            sys.stderr.reconfigure(encoding='utf-8')
        except Exception:
            pass

# 添加项目路径
project_root = Path(__file__).parent.absolute()
sys.path.insert(0, str(project_root))

from src.core.config import get_config
from src.services.embedding_service import EmbeddingService

def main():
    print("\n" + "=" * 80)
    print("重置熔断器状态")
    print("=" * 80)

    try:
        config = get_config()
        embedding_service = EmbeddingService(config_manager=config)

        # 获取熔断器状态
        before_state = embedding_service.circuit_breaker.get_state()
        print(f"\n[INFO] 重置前状态: {before_state}")

        # 重置熔断器
        embedding_service.circuit_breaker.reset()

        # 获取重置后状态
        after_state = embedding_service.circuit_breaker.get_state()
        print(f"[INFO] 重置后状态: {after_state}")

        print("\n[SUCCESS] 熔断器已重置，可以正常调用API")

    except Exception as e:
        print(f"\n[ERROR] 重置熔断器失败: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)

if __name__ == "__main__":
    main()
