#!/bin/bash

# Questions MCP Server 本地部署脚本
# 用于在 Linux/macOS 环境下部署服务

set -e

echo "=================================="
echo "Questions MCP Server 部署脚本"
echo "=================================="

# 颜色定义
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# 检查Python版本
check_python() {
    echo -e "${YELLOW}检查Python版本...${NC}"
    if ! command -v python3 &> /dev/null; then
        echo -e "${RED}错误: 未找到Python3${NC}"
        exit 1
    fi
    
    PYTHON_VERSION=$(python3 --version | cut -d' ' -f2 | cut -d'.' -f1,2)
    echo -e "${GREEN}✓ Python版本: $PYTHON_VERSION${NC}"
}

# 创建虚拟环境
create_venv() {
    echo -e "${YELLOW}创建Python虚拟环境...${NC}"
    if [ ! -d ".venv" ]; then
        python3 -m venv .venv
        echo -e "${GREEN}✓ 虚拟环境创建成功${NC}"
    else
        echo -e "${GREEN}✓ 虚拟环境已存在${NC}"
    fi
}

# 激活虚拟环境
activate_venv() {
    echo -e "${YELLOW}激活虚拟环境...${NC}"
    source .venv/bin/activate
    echo -e "${GREEN}✓ 虚拟环境已激活${NC}"
}

# 安装依赖
install_dependencies() {
    echo -e "${YELLOW}安装项目依赖...${NC}"
    pip install --upgrade pip
    pip install -r requirements.txt
    echo -e "${GREEN}✓ 依赖安装完成${NC}"
}

# 创建必要的目录
create_directories() {
    echo -e "${YELLOW}创建数据目录...${NC}"
    mkdir -p data/chroma
    mkdir -p data/backups
    mkdir -p logs
    echo -e "${GREEN}✓ 目录创建完成${NC}"
}

# 配置环境变量
setup_env() {
    echo -e "${YELLOW}配置环境变量...${NC}"
    if [ ! -f ".env" ]; then
        cp .env.example .env
        echo -e "${YELLOW}请编辑 .env 文件，配置必要的API密钥${NC}"
    else
        echo -e "${GREEN}✓ .env 文件已存在${NC}"
    fi
}

# 初始化数据库
init_database() {
    echo -e "${YELLOW}初始化数据库...${NC}"
    # 数据库会在首次运行时自动初始化
    echo -e "${GREEN}✓ 数据库将在首次启动时初始化${NC}"
}

# 主部署流程
main() {
    check_python
    create_venv
    activate_venv
    install_dependencies
    create_directories
    setup_env
    init_database
    
    echo -e "${GREEN}=================================="
    echo -e "部署完成！"
    echo -e "=================================="
    echo -e "运行服务: ${YELLOW}./start.sh${NC}"
    echo -e "停止服务: ${YELLOW}./stop.sh${NC}"
    echo -e "查看日志: ${YELLOW}tail -f logs/app.log${NC}"
}

main
