# MCP Server 题目管理与检索系统设计文档

## 1. 项目概述

### 1.1 项目目标

构建一个基于 FastMCP 框架的 MCP Server，实现题目的全生命周期管理、智能检索和统计分析功能。系统通过向量数据库存储题目的语义向量表示，利用 Embedding 大模型实现语义检索，同时提供完整的题目增删改查和多维度统计分析能力。

### 1.2 核心价值

- 支持题目的完整管理：创建、编辑、删除、批量导入
- 实现语义化检索，突破关键词匹配的局限性
- 提供多维度统计分析，洞察题库数据特征
- 通过标准化 MCP 协议接口，便于与各类 Agent 应用集成
- 支持 1-10 万级题目规模的高效存储和检索

### 1.3 应用场景

- AI 辅助学习系统的题目推荐与管理
- 智能题库管理平台的题目运营
- 教育领域 Agent 应用的题目服务中台
- 题目数据分析与质量评估

## 2. 系统架构

### 2.1 整体架构

系统采用三层架构设计，支持多种业务能力：

```mermaid
graph TB
    subgraph "客户端层"
        A[MCP Client/Agent应用]
    end
    
    subgraph "服务层 - FastMCP Server"
        B[MCP协议处理层]
        C[业务编排层]
        D1[检索服务]
        D2[管理服务]
        D3[统计服务]
        E[数据访问层]
    end
    
    subgraph "数据层"
        F[ChromaDB向量数据库]
        G[外部Embedding API]
        H[SQLite元数据库]
    end
    
    A -->|MCP协议| B
    B --> C
    C --> D1
    C --> D2
    C --> D3
    D1 --> E
    D2 --> E
    D3 --> E
    E -->|向量操作| F
    E -->|向量化请求| G
    E -->|元数据操作| H
```

### 2.2 核心组件

| 组件名称 | 职责描述 | 技术选型 |
|---------|---------|---------|
| MCP 协议处理层 | 处理 MCP 协议的请求解析和响应封装 | FastMCP 框架 |
| 业务编排层 | 协调各业务服务，处理跨服务的业务逻辑 | Python 业务代码 |
| 检索服务 | 实现语义检索和关键词检索功能 | Python + ChromaDB |
| 管理服务 | 实现题目的增删改查和批量操作 | Python |
| 统计服务 | 实现多维度数据统计和分析 | Python + SQL |
| 数据访问层 | 封装向量数据库和元数据库的操作 | Python SDK |
| 向量数据库 | 存储题目向量和基础元数据 | ChromaDB |
| 元数据库 | 存储题目详细信息和统计数据 | SQLite |
| Embedding 服务 | 提供文本向量化能力 | 云端 Embedding API |

### 2.3 架构设计考虑

**双数据库设计理由**：

- ChromaDB：专注于向量存储和相似度检索，性能优异
- SQLite：存储完整元数据、支持复杂查询和统计分析
- 两者数据通过 question_id 保持一致性

**服务分层理由**：

- 检索、管理、统计三个服务相对独立，便于维护和扩展
- 通过业务编排层统一协调，避免服务间直接依赖
- 数据访问层统一封装，确保数据一致性

## 3. 数据设计

### 3.1 题目数据模型

#### 3.1.1 完整数据模型

| 字段名称 | 数据类型 | 说明 | 是否必填 | 存储位置 |
|---------|---------|------|---------|---------|
| question_id | 字符串 | 题目唯一标识符（UUID） | 是 | 两者 |
| content | 文本 | 题目完整内容（题干+选项） | 是 | 两者 |
| title | 字符串 | 题目标题（简短描述） | 是 | SQLite |
| question_type | 字符串 | 题型（单选/多选/判断/填空/简答） | 是 | 两者 |
| category | 字符串 | 题目分类（如数学、英语等） | 是 | 两者 |
| difficulty | 字符串 | 难度等级（简单/中等/困难） | 是 | 两者 |
| tags | 字符串数组 | 题目标签 | 否 | 两者 |
| answer | 文本 | 参考答案 | 否 | SQLite |
| explanation | 文本 | 答案解析 | 否 | SQLite |
| source | 字符串 | 题目来源 | 否 | 两者 |
| source_url | 字符串 | 来源链接 | 否 | SQLite |
| points | 整数 | 题目分值 | 否 | SQLite |
| usage_count | 整数 | 使用次数 | 否 | SQLite |
| correct_rate | 浮点数 | 正确率（0-1） | 否 | SQLite |
| created_at | 时间戳 | 创建时间 | 是 | 两者 |
| updated_at | 时间戳 | 更新时间 | 是 | 两者 |
| created_by | 字符串 | 创建人 | 否 | SQLite |
| status | 字符串 | 状态（草稿/已发布/已归档） | 是 | 两者 |

#### 3.1.2 SQLite 数据库表设计

**questions 表**：

存储题目的完整信息，支持复杂查询和统计分析。

主键：question_id  
索引：category, difficulty, status, created_at

**tags 表**：

| 字段名 | 数据类型 | 说明 |
|-------|---------|------|
| tag_id | 整数 | 标签ID（自增） |
| tag_name | 字符串 | 标签名称 |
| usage_count | 整数 | 使用次数 |

**question_tags 关联表**：

| 字段名 | 数据类型 | 说明 |
|-------|---------|------|
| question_id | 字符串 | 题目ID |
| tag_id | 整数 | 标签ID |

### 3.2 向量存储结构

**ChromaDB Collection 配置**：

- **Collection 名称**：questions_collection
- **文档 ID**：使用 question_id 作为唯一标识
- **向量维度**：由所选 Embedding 模型决定（通常为 512-1536 维）
- **存储内容**：content 字段作为文档内容
- **元数据字段**：question_id, category, difficulty, question_type, status, tags, created_at

## 4. 接口设计

### 4.1 MCP Tools 定义

系统通过 MCP Server 暴露以下 Tools，按功能分类：

#### 4.1.1 检索类 Tools

##### search_questions

**功能描述**：根据查询文本进行语义检索

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 | 默认值 |
|-------|------|------|---------|--------|
| query | 字符串 | 用户查询文本 | 是 | - |
| top_k | 整数 | 返回结果数量 | 否 | 10 |
| category | 字符串 | 题目分类过滤 | 否 | null |
| difficulty | 字符串 | 难度等级过滤 | 否 | null |
| question_type | 字符串 | 题型过滤 | 否 | null |
| status | 字符串 | 状态过滤 | 否 | "已发布" |
| min_similarity | 浮点数 | 最低相似度阈值（0-1） | 否 | 0.0 |

**输出结果**：

返回题目列表，每个题目包含：
- question_id：题目ID
- title：题目标题
- content：题目内容
- similarity_score：相似度分数
- metadata：题目元数据

##### search_questions_by_keyword

**功能描述**：基于关键词的精确检索

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 | 默认值 |
|-------|------|------|---------|--------|
| keyword | 字符串 | 搜索关键词 | 是 | - |
| search_fields | 字符串数组 | 搜索字段（title/content/tags） | 否 | ["title", "content"] |
| limit | 整数 | 返回结果数量 | 否 | 20 |
| 其他过滤参数 | - | 与语义检索相同 | 否 | - |

**输出结果**：

返回匹配的题目列表

#### 4.1.2 管理类 Tools

##### create_question

**功能描述**：创建新题目

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 |
|-------|------|------|---------|
| title | 字符串 | 题目标题 | 是 |
| content | 文本 | 题目内容 | 是 |
| question_type | 字符串 | 题型 | 是 |
| category | 字符串 | 分类 | 是 |
| difficulty | 字符串 | 难度 | 是 |
| tags | 字符串数组 | 标签 | 否 |
| answer | 文本 | 参考答案 | 否 |
| explanation | 文本 | 答案解析 | 否 |
| source | 字符串 | 来源 | 否 |
| points | 整数 | 分值 | 否 |
| status | 字符串 | 状态 | 否（默认"草稿"） |

**输出结果**：

- question_id：新创建的题目ID
- success：是否成功
- message：操作消息

##### update_question

**功能描述**：更新题目信息

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 |
|-------|------|------|---------|
| question_id | 字符串 | 题目ID | 是 |
| 其他字段 | - | 需要更新的字段 | 至少一个 |

**输出结果**：

- success：是否成功
- message：操作消息

**注意事项**：

- 更新 content 字段会触发重新向量化
- 更新 updated_at 字段自动更新为当前时间

##### delete_question

**功能描述**：删除题目

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 |
|-------|------|------|---------|
| question_id | 字符串 | 题目ID | 是 |
| soft_delete | 布尔值 | 是否软删除（改状态为已归档） | 否（默认false） |

**输出结果**：

- success：是否成功
- message：操作消息

##### get_question

**功能描述**：获取单个题目详情

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 |
|-------|------|------|---------|
| question_id | 字符串 | 题目ID | 是 |
| include_answer | 布尔值 | 是否包含答案 | 否（默认true） |

**输出结果**：

返回题目完整信息

##### list_questions

**功能描述**：获取题目列表（分页）

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 | 默认值 |
|-------|------|------|---------|--------|
| page | 整数 | 页码（从1开始） | 否 | 1 |
| page_size | 整数 | 每页数量 | 否 | 20 |
| category | 字符串 | 分类过滤 | 否 | null |
| difficulty | 字符串 | 难度过滤 | 否 | null |
| question_type | 字符串 | 题型过滤 | 否 | null |
| status | 字符串 | 状态过滤 | 否 | null |
| sort_by | 字符串 | 排序字段 | 否 | "created_at" |
| sort_order | 字符串 | 排序方向（asc/desc） | 否 | "desc" |

**输出结果**：

- total：总数量
- page：当前页码
- page_size：每页数量
- items：题目列表

##### import_questions

**功能描述**：批量导入题目（从外部API）

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 |
|-------|------|------|---------|
| api_endpoint | 字符串 | 外部API地址 | 否（使用配置） |
| batch_size | 整数 | 每批处理数量 | 否（使用配置） |
| category_filter | 字符串 | 只导入指定分类 | 否 |
| skip_duplicates | 布尔值 | 跳过重复题目 | 否（默认true） |

**输出结果**：

- total_imported：成功导入数量
- total_skipped：跳过数量
- total_failed：失败数量
- errors：错误列表

#### 4.1.3 统计分析类 Tools

##### get_statistics_overview

**功能描述**：获取题库整体统计数据

**输入参数**：无

**输出结果**：

- total_questions：题目总数
- by_category：各分类题目数量分布
- by_difficulty：各难度题目数量分布
- by_question_type：各题型数量分布
- by_status：各状态数量分布
- average_usage_count：平均使用次数
- average_correct_rate：平均正确率

##### get_category_statistics

**功能描述**：获取指定分类的统计数据

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 |
|-------|------|------|---------|
| category | 字符串 | 题目分类 | 是 |

**输出结果**：

- category：分类名称
- total_count：题目总数
- by_difficulty：难度分布
- by_question_type：题型分布
- top_tags：高频标签（Top 10）
- average_correct_rate：平均正确率
- trend：增长趋势（按月统计近6个月）

##### get_tag_statistics

**功能描述**：获取标签使用统计

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 |
|-------|------|------|---------|
| top_n | 整数 | 返回Top N标签 | 否（默认20） |

**输出结果**：

返回标签列表，每个标签包含：
- tag_name：标签名称
- usage_count：使用次数
- categories：关联的分类

##### get_quality_report

**功能描述**：生成题库质量报告

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 |
|-------|------|------|---------|
| category | 字符串 | 指定分类（可选） | 否 |

**输出结果**：

- total_evaluated：已评估题目数（有正确率数据）
- high_quality_count：高质量题目数（正确率0.3-0.7）
- low_quality_count：低质量题目数（正确率<0.2或>0.9）
- no_usage_count：未使用题目数
- recommendations：改进建议列表

##### get_time_series_statistics

**功能描述**：获取时间序列统计数据

**输入参数**：

| 参数名 | 类型 | 说明 | 是否必填 |
|-------|------|------|---------|
| time_range | 字符串 | 时间范围（7d/30d/90d/1y） | 否（默认30d） |
| granularity | 字符串 | 粒度（day/week/month） | 否（默认day） |
| metric | 字符串 | 指标（count/usage） | 否（默认count） |

**输出结果**：

- time_series：时间序列数据数组
  - date：日期
  - value：指标值
- total：总计
- growth_rate：增长率

## 5. 核心流程设计

### 5.1 题目创建流程

```mermaid
flowchart TD
    A[接收创建请求] --> B[参数验证]
    B --> C{验证通过?}
    C -->|否| D[返回参数错误]
    C -->|是| E[生成question_id]
    E --> F[调用Embedding API向量化content]
    F --> G{向量化成功?}
    G -->|否| H[返回向量化错误]
    G -->|是| I[开始数据库事务]
    I --> J[写入SQLite数据库]
    J --> K[写入ChromaDB]
    K --> L{两者都成功?}
    L -->|否| M[回滚事务]
    M --> N[返回存储错误]
    L -->|是| O[提交事务]
    O --> P[更新标签统计]
    P --> Q[返回成功结果]
```

**关键策略**：

- 使用数据库事务确保两个数据库的数据一致性
- 向量化失败不影响数据写入，可后续补偿
- 自动处理标签的创建和关联

### 5.2 题目更新流程

```mermaid
flowchart TD
    A[接收更新请求] --> B[验证question_id存在性]
    B --> C{题目存在?}
    C -->|否| D[返回不存在错误]
    C -->|是| E{是否更新content?}
    E -->|是| F[调用Embedding API重新向量化]
    E -->|否| G[跳过向量化]
    F --> G
    G --> H[开始数据库事务]
    H --> I[更新SQLite数据]
    I --> J{需要更新向量?}
    J -->|是| K[更新ChromaDB向量和元数据]
    J -->|否| L[仅更新ChromaDB元数据]
    K --> M{两者都成功?}
    L --> M
    M -->|否| N[回滚事务]
    N --> O[返回更新错误]
    M -->|是| P[提交事务]
    P --> Q[更新updated_at时间戳]
    Q --> R[返回成功结果]
```

**关键策略**：

- 只有 content 字段变更才触发重新向量化
- 元数据更新同步到两个数据库
- 保持数据一致性，任一失败则全部回滚

### 5.3 题目删除流程

```mermaid
flowchart TD
    A[接收删除请求] --> B{是软删除?}
    B -->|是| C[将status改为已归档]
    C --> D[更新SQLite]
    D --> E[更新ChromaDB元数据]
    E --> F[返回成功]
    B -->|否| G[开始数据库事务]
    G --> H[从SQLite删除]
    H --> I[从question_tags删除关联]
    I --> J[从ChromaDB删除]
    J --> K{两者都成功?}
    K -->|否| L[回滚事务]
    L --> M[返回删除错误]
    K -->|是| N[提交事务]
    N --> O[更新标签使用计数]
    O --> F
```

**关键策略**：

- 支持软删除（状态归档）和硬删除两种模式
- 硬删除需清理关联数据（标签关联等）
- 删除后更新相关统计数据

### 5.4 题目检索流程

```mermaid
flowchart TD
    A[接收检索请求] --> B[参数验证]
    B --> C{参数有效?}
    C -->|否| D[返回参数错误]
    C -->|是| E[调用Embedding API向量化query]
    E --> F{向量化成功?}
    F -->|否| G[返回向量化错误]
    F -->|是| H[构建元数据过滤条件]
    H --> I[向ChromaDB发起相似度检索]
    I --> J[应用相似度阈值过滤]
    J --> K[取top_k结果]
    K --> L{需要完整信息?}
    L -->|是| M[根据question_id从SQLite查询]
    L -->|否| N[直接返回ChromaDB结果]
    M --> O[合并两者数据]
    O --> P[返回检索结果]
    N --> P
```

**关键策略**：

- 先在 ChromaDB 进行向量检索，性能最优
- 利用 ChromaDB 的元数据过滤能力预筛选
- 需要完整信息时才查询 SQLite，减少数据库压力
- 支持相似度阈值过滤低质量结果

### 5.5 批量导入流程

```mermaid
flowchart TD
    A[开始导入] --> B[调用外部API获取题目数据]
    B --> C{API调用成功?}
    C -->|否| D[记录错误并重试]
    D --> E{重试次数<3?}
    E -->|是| B
    E -->|否| F[跳过此批次]
    C -->|是| G[解析题目数据]
    G --> H[按batch_size分批]
    H --> I[检查重复题目]
    I --> J[过滤重复项]
    J --> K[批量调用Embedding API]
    K --> L{向量化成功?}
    L -->|否| D
    L -->|是| M[开始批量写入事务]
    M --> N[批量写入SQLite]
    N --> O[批量写入ChromaDB]
    O --> P{两者都成功?}
    P -->|否| Q[回滚此批次]
    Q --> R[记录失败日志]
    P -->|是| S[提交事务]
    S --> T[更新导入进度]
    T --> U{还有更多数据?}
    U -->|是| B
    U -->|否| V[生成导入报告]
    V --> W[导入完成]
    F --> W
    R --> W
```

**关键策略**：

- 批量处理，每批 50-100 条题目，提高效率
- 重复检测基于题目内容相似度或唯一字段
- 失败重试机制，最多重试 3 次
- 记录详细的导入日志和报告
- 支持断点续传，记录已导入的位置

### 5.6 统计数据生成流程

```mermaid
flowchart TD
    A[接收统计请求] --> B[解析统计类型和参数]
    B --> C{统计类型?}
    C -->|整体概览| D[查询SQLite汇总数据]
    C -->|分类统计| E[按分类聚合查询]
    C -->|标签统计| F[关联查询tags和question_tags]
    C -->|质量报告| G[分析正确率和使用数据]
    C -->|时间序列| H[按时间维度分组查询]
    D --> I[计算派生指标]
    E --> I
    F --> I
    G --> I
    H --> I
    I --> J[格式化输出结果]
    J --> K[返回统计数据]
```

**关键策略**：

- 统计数据主要基于 SQLite，利用其强大的 SQL 分析能力
- 使用聚合函数和分组查询提高效率
- 对频繁访问的统计数据考虑缓存机制
- 复杂统计可异步生成，提供查询进度接口

## 6. 技术方案

### 6.1 FastMCP 框架集成

**框架特性利用**：

- 使用 `@mcp.tool()` 装饰器定义所有 MCP Tools
- 利用框架的参数验证和类型检查机制
- 实现统一的错误处理和响应格式化
- 遵循 MCP 协议规范

**Server 初始化**：

- 在启动时初始化 ChromaDB 和 SQLite 连接
- 预加载配置信息（API密钥、数据库路径等）
- 检查数据库表结构，自动创建缺失的表
- 实现优雅启动和关闭

### 6.2 ChromaDB 配置

**部署模式**：

- 本地持久化模式，数据目录可配置
- 适合 1-10 万级题目规模

**Collection 配置**：

- 距离度量：余弦相似度（cosine）
- 索引类型：HNSW
- 元数据索引字段：category, difficulty, question_type, status

**性能优化**：

- 批量操作使用 `add()` 的批量模式
- 启用向量压缩
- 设置合理的 HNSW 参数（ef_construction, M）

### 6.3 SQLite 配置

**数据库设计**：

- 单文件数据库，便于备份和迁移
- 启用 WAL 模式提高并发性能
- 创建必要的索引加速查询

**连接池管理**：

- 使用 Python 的 sqlite3 连接池
- 设置合理的连接超时时间
- 实现连接的自动重连机制

**事务管理**：

- 写操作使用事务确保一致性
- 读操作使用共享锁提高并发
- 实现两个数据库的分布式事务协调

### 6.4 Embedding API 集成

**API 选择建议**：

| API 服务商 | 模型示例 | 向量维度 | 中文能力 | 适用场景 |
|-----------|---------|---------|---------|---------|
| OpenAI | text-embedding-3-small | 1536 | 良好 | 通用场景 |
| 通义千问 | text-embedding-v2 | 1536 | 优秀 | 中文为主 |
| 智谱AI | embedding-2 | 1024 | 优秀 | 性价比优先 |

**调用策略**：

- 实现 API 调用的重试和熔断机制
- 批量调用接口，每批最多 20-50 条
- 设置超时时间（建议 30 秒）
- 实现常见查询的向量缓存（使用 LRU 策略）

**成本优化**：

- 缓存高频查询的向量
- 批量处理减少调用次数
- 监控 API 用量，设置告警阈值

### 6.5 数据一致性保证

**两阶段提交策略**：

- 先写 SQLite（主数据），再写 ChromaDB（向量）
- SQLite 失败则整体失败
- ChromaDB 失败则回滚 SQLite 并返回错误

**数据同步检查**：

- 定期扫描两个数据库的 question_id 集合
- 发现不一致则记录日志并触发告警
- 提供数据修复工具

**失败补偿机制**：

- 记录向量化失败的题目 ID
- 提供后台任务重新向量化
- 允许手动触发数据同步

### 6.6 错误处理机制

**异常分类**：

| 异常类型 | HTTP状态码 | 处理策略 | 用户提示 |
|---------|-----------|---------|---------|
| 参数验证失败 | 400 | 直接返回 | 具体参数错误信息 |
| 题目不存在 | 404 | 直接返回 | "题目不存在" |
| Embedding API 失败 | 503 | 重试3次 | "向量化服务暂时不可用" |
| 数据库写入失败 | 500 | 回滚事务 | "数据保存失败，请重试" |
| 数据库查询失败 | 500 | 记录日志 | "查询失败，请稍后重试" |

**错误日志记录**：

- 记录完整的请求参数和堆栈信息
- 包含 request_id 便于追踪
- 敏感信息（如API密钥）需脱敏

## 7. 配置管理

### 7.1 配置项定义

| 配置项 | 说明 | 示例值 | 配置方式 |
|-------|------|-------|---------|
| EMBEDDING_API_KEY | Embedding API 认证密钥 | sk-xxx | 环境变量 |
| EMBEDDING_API_ENDPOINT | API 端点地址 | https://api.example.com/v1/embeddings | 配置文件 |
| EMBEDDING_MODEL | 模型名称 | text-embedding-v2 | 配置文件 |
| CHROMA_PERSIST_DIR | ChromaDB 持久化目录 | ./data/chroma | 配置文件 |
| CHROMA_COLLECTION_NAME | Collection 名称 | questions_collection | 配置文件 |
| SQLITE_DB_PATH | SQLite 数据库文件路径 | ./data/questions.db | 配置文件 |
| EXTERNAL_API_ENDPOINT | 外部题目API地址 | https://questions-api.example.com | 配置文件 |
| EXTERNAL_API_KEY | 外部API密钥 | api-key-xxx | 环境变量 |
| BATCH_SIZE | 批量处理大小 | 50 | 配置文件 |
| MAX_RETRIES | 最大重试次数 | 3 | 配置文件 |
| EMBEDDING_CACHE_SIZE | 向量缓存大小 | 1000 | 配置文件 |
| LOG_LEVEL | 日志级别 | INFO | 环境变量 |

### 7.2 配置文件结构

使用 YAML 格式的配置文件（config.yaml）：

- server 部分：MCP Server 相关配置
- database 部分：数据库连接配置
- embedding 部分：Embedding API 配置
- import 部分：导入相关配置
- cache 部分：缓存配置

### 7.3 配置加载与验证

**加载顺序**：

- 加载默认配置
- 加载配置文件覆盖默认值
- 加载环境变量覆盖配置文件

**启动验证**：

- 检查必需配置项是否存在
- 验证配置值的合法性（如路径是否可写）
- 测试外部服务连接性（Embedding API）

## 8. 实施计划

### 8.1 开发阶段划分

项目分为 8 个开发阶段：

#### 阶段 1：环境准备与基础框架搭建 ✅ 已完成

**目标**：建立项目基础架构

**关键任务**：
- ✅ 创建 Python 项目和虚拟环境
- ✅ 安装核心依赖（fastmcp, chromadb, sqlite3, requests等）
- ✅ 搭建项目目录结构（按服务分层）
- ✅ 实现配置管理模块
- ✅ 实现日志记录模块
- ✅ 编写工具函数（UUID生成、时间处理等）

**交付物**：
- ✅ 可运行的项目骨架
- ✅ 配置文件模板（config.yaml）
- ✅ requirements.txt
- ⏳ .env.example（待补充）

**实际完成时间**：已完成

**完成情况说明**：
- 项目目录结构完整，包含 config/、src/core/、src/database/、src/services/、src/utils/ 等目录
- 配置管理模块（src/core/config.py）已实现，支持 YAML 文件和环境变量配置，包含完整的配置验证机制
- 日志管理模块（src/core/logger.py）已实现，支持文件日志、控制台日志、敏感信息脱敏、request_id 追踪
- 工具函数模块（src/utils/helpers.py）已实现，包含 UUID 生成、时间处理、数据验证等常用函数
- 主程序入口（src/main.py）已搭建，包含 FastMCP Server 初始化、健康检查等基础 Tools
- requirements.txt 已完善，包含所有核心依赖
- config.yaml 配置文件详尽，涵盖所有模块的配置项

#### 阶段 2：数据库设计与初始化 ⏳ 进行中

**目标**：完成两个数据库的设计和初始化

**关键任务**：
- ⏳ 设计 SQLite 数据库表结构
- ⏳ 编写数据库初始化脚本（建表、建索引）
- ⏳ 初始化 ChromaDB Collection
- ⏳ 实现数据访问层（DAO）的基础封装
- ⏳ 实现数据库连接管理和连接池
- ⏳ 编写数据库操作的单元测试

**交付物**：
- ⏳ SQLite 数据库 schema 脚本
- ⏳ ChromaDB 初始化脚本
- ⏳ 数据访问层基础模块（src/database/ 目录已创建，待实现）

**预估时间**：1 天

**当前状态**：数据库模块目录结构已创建，核心文件待实现

#### 阶段 3：Embedding API 集成 ⏸️ 未开始

**目标**：实现文本向量化能力

**关键任务**：
- 选择并配置 Embedding API
- 实现 API 调用封装
- 实现批量向量化接口
- 实现重试和熔断机制
- 实现向量缓存（LRU）
- 编写 API 调用测试

**交付物**：
- Embedding API 封装模块
- 缓存管理模块
- API 调用测试报告

**预估时间**：1 天

**依赖**：阶段 1 已完成

#### 阶段 4：题目管理服务实现 ⏸️ 未开始

**目标**：实现题目的完整管理功能

**关键任务**：
- 实现 create_question（创建题目）
- 实现 update_question（更新题目）
- 实现 delete_question（删除题目）
- 实现 get_question（获取详情）
- 实现 list_questions（分页列表）
- 实现数据一致性保证（两阶段提交）
- 实现标签管理逻辑
- 编写管理服务的单元测试

**交付物**：
- 题目管理服务模块
- 标签管理模块
- 单元测试覆盖

**预估时间**：2 天

**依赖**：阶段 2、阶段 3

#### 阶段 5：题目检索服务实现 ⏸️ 未开始

**目标**：实现语义检索和关键词检索

**关键任务**：
- 实现 search_questions（语义检索）
- 实现 search_questions_by_keyword（关键词检索）
- 实现查询预处理逻辑
- 实现结果排序和过滤
- 实现检索结果缓存
- 优化检索性能
- 编写检索服务测试

**交付物**：
- 题目检索服务模块
- 检索性能测试报告

**预估时间**：1.5 天

**依赖**：阶段 2、阶段 3

#### 阶段 6：批量导入功能实现 ⏸️ 未开始

**目标**：实现从外部API批量导入题目

**关键任务**：
- 实现 import_questions Tool
- 对接外部题目数据源 API
- 实现数据解析和转换
- 实现批量导入流程
- 实现重复检测逻辑
- 实现导入进度跟踪
- 实现失败重试和断点续传
- 生成导入报告

**交付物**：
- 批量导入模块
- 导入测试数据集
- 导入功能验证报告

**预估时间**：1.5 天

**依赖**：阶段 4、阶段 5

#### 阶段 7：统计分析服务实现 ⏸️ 未开始

**目标**：实现多维度统计分析功能

**关键任务**：
- 实现 get_statistics_overview（整体概览）
- 实现 get_category_statistics（分类统计）
- 实现 get_tag_statistics（标签统计）
- 实现 get_quality_report（质量报告）
- 实现 get_time_series_statistics（时间序列）
- 优化 SQL 查询性能
- 考虑统计数据缓存策略
- 编写统计服务测试

**交付物**：
- 统计分析服务模块
- 统计报告样例

**预估时间**：1.5 天

**依赖**：阶段 2、阶段 4

#### 阶段 8：FastMCP Server 集成与测试 ⏸️ 未开始

**目标**：集成所有服务到 FastMCP Server，完成端到端测试

**关键任务**：
- 使用 FastMCP 框架定义所有 MCP Tools
- 实现业务编排层
- 实现统一的参数验证
- 实现统一的错误处理
- 编写端到端集成测试
- 进行性能测试和优化
- 编写用户使用文档
- 准备部署脚本

**交付物**：
- 完整的 MCP Server 应用
- 集成测试报告
- 性能测试报告
- 用户文档
- 部署文档

**预估时间**：1.5 天

**依赖**：阶段 4、阶段 5、阶段 6、阶段 7

### 8.2 阶段依赖关系

```mermaid
graph LR
    A[阶段1: 基础框架] --> B[阶段2: 数据库]
    A --> C[阶段3: Embedding API]
    B --> D[阶段4: 管理服务]
    C --> D
    B --> E[阶段5: 检索服务]
    C --> E
    D --> F[阶段6: 批量导入]
    E --> F
    B --> G[阶段7: 统计服务]
    D --> H[阶段8: 集成测试]
    E --> H
    F --> H
    G --> H
```

### 8.3 总体时间估算

| 阶段 | 预估工作量 | 关键风险 |
|-----|----------|---------|
| 阶段 1 | 0.5 天 | 依赖包版本冲突 |
| 阶段 2 | 1 天 | 数据库设计复杂度 |
| 阶段 3 | 1 天 | API 选择与调试 |
| 阶段 4 | 2 天 | 数据一致性实现难度 |
| 阶段 5 | 1.5 天 | 检索性能优化 |
| 阶段 6 | 1.5 天 | 外部 API 稳定性 |
| 阶段 7 | 1.5 天 | SQL 查询优化 |
| 阶段 8 | 1.5 天 | 集成问题排查 |
| **总计** | **10.5 天** | - |

## 9. 风险与应对

### 9.1 技术风险

| 风险项 | 影响程度 | 概率 | 应对策略 |
|-------|---------|------|---------|
| Embedding API 调用频率限制 | 高 | 中 | 实现请求限流、批量处理、多 API 切换机制 |
| 两个数据库数据不一致 | 高 | 中 | 实现事务协调、定期一致性检查、提供修复工具 |
| ChromaDB 性能瓶颈 | 中 | 低 | 优化索引参数、考虑分 Collection 存储 |
| SQLite 并发写入冲突 | 中 | 中 | 启用 WAL 模式、合理设置超时时间 |
| 外部数据源 API 不稳定 | 中 | 中 | 实现重试机制、记录失败数据、支持断点续传 |
| 向量检索准确性不足 | 中 | 低 | 调整相似度阈值、尝试不同 Embedding 模型 |
| 统计查询性能差 | 中 | 低 | 优化 SQL 索引、引入缓存机制 |

### 9.2 成本风险

| 风险项 | 影响程度 | 应对策略 |
|-------|---------|---------|
| Embedding API 调用成本超预算 | 中 | 实现向量缓存、选择性价比高的服务商、监控用量 |
| 数据存储成本增长 | 低 | 定期清理归档数据、启用向量压缩 |

### 9.3 业务风险

| 风险项 | 影响程度 | 应对策略 |
|-------|---------|---------|
| 导入数据质量差 | 中 | 实现数据验证规则、提供数据清洗工具 |
| 题目重复率高 | 低 | 实现智能去重（基于相似度） |
| 用户误删重要数据 | 中 | 默认软删除、提供数据恢复功能、定期备份 |

## 10. 扩展性考虑

### 10.1 功能扩展方向

系统设计预留以下扩展可能：

**权限管理**：
- 支持用户角色和权限控制
- 不同角色对题目的操作权限不同
- 支持题目的审核流程

**多模态支持**：
- 支持图片题目的存储和检索
- 支持公式识别和向量化
- 支持音频、视频题目

**智能推荐**：
- 基于学习记录的个性化题目推荐
- 基于错题的薄弱点分析
- 自适应难度调整

**协作功能**：
- 支持多人协作编辑题目
- 题目版本管理
- 评论和讨论功能

**导出功能**：
- 支持导出为各种格式（Word、PDF、Excel）
- 支持按条件批量导出
- 支持组卷功能

### 10.2 性能扩展策略

**数据规模超过 10 万时**：

- 迁移到分布式向量数据库（如 Milvus）
- SQLite 替换为 PostgreSQL 或 MySQL
- 引入 Redis 缓存层
- 实现数据分片策略

**高并发场景**：

- 引入消息队列处理批量操作
- 实现读写分离
- 部署多实例负载均衡
- 使用 CDN 缓存静态资源

**数据备份与恢复**：

- 实现自动备份机制（每日备份）
- 提供数据恢复工具
- 支持增量备份

## 11. 监控与运维

### 11.1 关键指标监控

| 指标类别 | 指标名称 | 监控目的 | 告警阈值建议 |
|---------|---------|---------|------------|
| 可用性 | MCP Server 健康状态 | 确保服务可用 | 服务停止 |
| 可用性 | 数据库连接状态 | 确保数据层可用 | 连接失败 |
| 性能 | 平均检索响应时间 | 评估检索性能 | 超过 2 秒 |
| 性能 | 平均创建响应时间 | 评估写入性能 | 超过 3 秒 |
| 性能 | Embedding API 响应时间 | 监控外部依赖 | 超过 5 秒 |
| 业务 | 题目总数 | 监控数据规模 | 接近容量上限 |
| 业务 | 每日新增题目数 | 监控数据增长 | 异常增长/减少 |
| 业务 | 检索请求量 | 监控使用情况 | - |
| 成本 | Embedding API 调用次数 | 控制成本 | 超过预算 80% |
| 质量 | API 调用成功率 | 评估稳定性 | 低于 95% |
| 质量 | 数据一致性检查 | 确保数据完整 | 发现不一致 |

### 11.2 日志记录策略

**日志级别**：

- ERROR：严重错误（API 失败、数据库异常、数据不一致等）
- WARNING：警告信息（重试操作、配置缺失、性能慢等）
- INFO：关键流程（请求处理、导入进度、统计生成等）
- DEBUG：调试信息（详细参数、中间结果等，生产环境关闭）

**日志内容**：

- 时间戳、日志级别、模块名称
- request_id（追踪完整请求链路）
- 用户标识（如有）
- 关键参数和执行结果
- 异常堆栈信息
- 执行耗时

**日志管理**：

- 日志文件按日期滚动
- 保留最近 30 天日志
- 错误日志单独存储
- 敏感信息脱敏处理

### 11.3 运维工具

**数据一致性检查工具**：

- 定期扫描两个数据库的 question_id
- 生成一致性检查报告
- 提供自动修复功能

**数据备份工具**：

- 自动备份 SQLite 数据库
- 自动备份 ChromaDB 数据目录
- 支持手动触发备份
- 验证备份文件完整性

**性能分析工具**：

- 记录慢查询日志
- 分析 API 调用耗时分布
- 生成性能分析报告

**数据导出工具**：

- 支持按条件导出题目数据
- 支持多种导出格式
- 支持大批量数据导出

## 12. 安全考虑

### 12.1 数据安全

**敏感信息保护**：

- API 密钥使用环境变量存储
- 数据库连接信息加密存储
- 日志中脱敏敏感字段

**数据访问控制**：

- 验证所有输入参数，防止 SQL 注入
- 限制查询结果数量，防止数据泄露
- 题目答案可选择性返回

**数据备份**：

- 定期备份数据库
- 备份文件加密存储
- 测试备份恢复流程

### 12.2 接口安全

**认证与授权**：

- MCP Server 支持认证机制
- 不同操作需要不同权限
- 记录操作审计日志

**请求限流**：

- 防止恶意请求耗尽资源
- 按 IP 或用户限制请求频率
- 批量操作限制最大数量

**输入验证**：

- 验证所有参数类型和范围
- 防止超长文本导致性能问题
- 过滤恶意输入内容

### 12.3 运行安全

**异常处理**：

- 不向外暴露内部错误细节
- 记录完整错误信息用于排查
- 实现优雅降级机制

**资源限制**：

- 限制单次查询返回数量
- 限制批量操作的批次大小
- 设置数据库查询超时时间

**依赖安全**：

- 定期更新依赖包版本
- 扫描依赖包安全漏洞
- 使用可信的依赖源

## 13. 当前项目实现状态

### 13.1 项目总体进度

**当前阶段**：阶段 1 已完成，阶段 2 进行中

**完成度**：约 10%（8 个阶段中的第 1 个阶段已完成）

**项目状态图示**：

```mermaid
gantt
    title MCP Server 开发进度
    dateFormat YYYY-MM-DD
    section 已完成
    阶段1 环境准备与基础框架 :done, stage1, 2024-01-01, 1d
    section 进行中
    阶段2 数据库设计与初始化 :active, stage2, 2024-01-02, 1d
    section 待开始
    阶段3 Embedding API集成 :stage3, after stage2, 1d
    阶段4 题目管理服务 :stage4, after stage3, 2d
    阶段5 题目检索服务 :stage5, after stage3, 1.5d
    阶段6 批量导入功能 :stage6, after stage4, 1.5d
    阶段7 统计分析服务 :stage7, after stage4, 1.5d
    阶段8 集成与测试 :stage8, after stage6, 1.5d
```

### 13.2 已完成模块详情

#### 13.2.1 项目目录结构

当前项目目录组织清晰，按功能模块划分：

```
questions_mcp_server/
├── config/
│   └── config.yaml          # ✅ 完整的配置文件
├── src/
│   ├── core/
│   │   ├── __init__.py
│   │   ├── config.py        # ✅ 配置管理模块（完善）
│   │   └── logger.py        # ✅ 日志管理模块（完善）
│   ├── database/
│   │   └── __init__.py      # ⏳ 数据库模块（待实现）
│   ├── services/
│   │   └── __init__.py      # ⏸️ 业务服务（待实现）
│   ├── utils/
│   │   ├── __init__.py
│   │   └── helpers.py       # ✅ 工具函数模块（完善）
│   ├── __init__.py
│   └── main.py              # ✅ 主程序入口（基础完成）
└── requirements.txt         # ✅ 依赖清单
```

#### 13.2.2 配置管理模块（src/core/config.py）

**实现功能**：
- ✅ 支持 YAML 配置文件加载
- ✅ 支持环境变量配置覆盖
- ✅ 配置分层管理（server、database、embedding、import、cache、logging 等）
- ✅ 配置验证机制（必需项检查、路径验证、数值范围验证）
- ✅ 自动创建配置目录（数据库目录、日志目录等）
- ✅ 单例模式配置实例
- ✅ 提供便捷属性访问（embedding_api_key、sqlite_db_path 等）

**配置覆盖优先级**：环境变量 > YAML 配置文件 > 默认值

**核心特性**：
- 嵌套配置支持（如 `database.sqlite.db_path`）
- 配置热加载（reload_config 方法）
- 完善的错误处理和提示

#### 13.2.3 日志管理模块（src/core/logger.py）

**实现功能**：
- ✅ 多输出目标（文件日志、控制台日志、错误日志单独存储）
- ✅ 日志轮转（基于文件大小，保留指定数量的备份）
- ✅ 日志级别控制（DEBUG、INFO、WARNING、ERROR、CRITICAL）
- ✅ request_id 追踪（线程本地存储，追踪完整请求链路）
- ✅ 敏感信息脱敏（API 密钥、密码、Token 等自动脱敏）
- ✅ 自定义日志格式
- ✅ 单例模式日志实例

**核心特性**：
- 自动创建日志目录
- 支持独立的错误日志文件
- 敏感信息正则匹配脱敏
- 便捷的全局日志函数（info、error、warning 等）

#### 13.2.4 工具函数模块（src/utils/helpers.py）

**实现功能**：
- ✅ UUID 生成（完整 UUID、短 ID）
- ✅ 时间处理（获取当前时间戳、时间格式化、时间转换）
- ✅ 哈希计算（MD5、SHA256）
- ✅ 字符串处理（清理、截断、验证）
- ✅ 数据验证（邮箱、URL 格式验证）
- ✅ 字典操作（安全获取、合并）
- ✅ 列表操作（分块、扁平化、去重）
- ✅ 查询字符串处理
- ✅ 文件大小格式化
- ✅ 空值检查和合并
- ✅ 错误消息格式化

**核心特性**：
- 函数设计遵循单一职责原则
- 完善的类型注解
- 健壮的异常处理

#### 13.2.5 主程序入口（src/main.py）

**实现功能**：
- ✅ 系统初始化流程（配置加载、日志初始化、配置验证）
- ✅ FastMCP Server 创建
- ✅ 基础 MCP Tools 注册
  - `health_check`：健康检查
  - `get_server_info`：获取服务器信息
  - `test_configuration`：测试配置
- ✅ 优雅启动和关闭（信号处理）
- ✅ 友好的命令行输出

**核心特性**：
- 清晰的启动流程
- 完善的错误处理
- 信号量处理（SIGINT、SIGTERM）
- 启动过程可视化反馈

#### 13.2.6 配置文件（config/config.yaml）

**已配置模块**：
- ✅ Server 配置（名称、版本、描述、端口）
- ✅ Database 配置（SQLite、ChromaDB 详细配置）
- ✅ Embedding API 配置（端点、模型、批量处理、重试、熔断）
- ✅ Import 配置（外部 API、批量处理、重复检测、断点续传）
- ✅ Cache 配置（向量缓存、查询缓存、统计缓存）
- ✅ Logging 配置（级别、格式、文件轮转）
- ✅ Performance 配置（查询、分页、连接池）
- ✅ Security 配置（限流、输入验证、敏感字段）
- ✅ Data Management 配置（备份、清理、一致性检查）
- ✅ Monitoring 配置（健康检查、指标、告警）
- ✅ Features 配置（功能开关）

**特点**：配置项完整、注释清晰、结构合理

#### 13.2.7 依赖管理（requirements.txt）

**已声明依赖**：
- ✅ FastMCP 框架（>=0.2.0）
- ✅ ChromaDB（>=0.4.0）
- ✅ Requests（>=2.31.0）
- ✅ Pydantic（>=2.5.0）
- ✅ Python-dotenv（>=1.0.0）
- ✅ PyYAML（>=6.0.1）
- ✅ Python-dateutil（>=2.8.2）
- ✅ UUID
- ✅ 测试工具（pytest、pytest-asyncio）
- ✅ 代码质量工具（black、flake8）

### 13.3 待实现模块清单

#### 13.3.1 阶段 2 待实现内容（进行中）

**数据库模块（src/database/）**：

1. **数据模型定义**（models.py）
   - QuestionModel：题目数据模型
   - TagModel：标签数据模型
   - 数据验证规则

2. **SQLite 数据访问层**（sqlite_dao.py）
   - 数据库连接管理
   - 表结构初始化
   - CRUD 操作封装
   - 事务管理
   - 标签关联操作

3. **ChromaDB 数据访问层**（chroma_dao.py）
   - Collection 初始化
   - 向量存储操作
   - 相似度检索
   - 元数据过滤
   - 批量操作

4. **数据库管理器**（database_manager.py）
   - 协调两个数据库操作
   - 两阶段提交实现
   - 数据一致性检查
   - 失败补偿机制

5. **数据库初始化脚本**（schema.sql）
   - questions 表定义
   - tags 表定义
   - question_tags 关联表定义
   - 索引创建

6. **环境变量示例文件**（.env.example）
   - API 密钥模板
   - 数据库路径配置
   - 其他敏感配置项

#### 13.3.2 后续阶段待实现内容

**阶段 3 - Embedding API 集成**：
- Embedding API 调用封装（src/services/embedding_service.py）
- 批量向量化接口
- 重试和熔断机制
- LRU 缓存实现（src/utils/cache.py）

**阶段 4 - 题目管理服务**：
- 题目管理服务（src/services/question_service.py）
- 标签管理服务（src/services/tag_service.py）
- MCP Tools 实现（create_question、update_question、delete_question、get_question、list_questions）

**阶段 5 - 题目检索服务**：
- 检索服务（src/services/search_service.py）
- MCP Tools 实现（search_questions、search_questions_by_keyword）

**阶段 6 - 批量导入功能**：
- 导入服务（src/services/import_service.py）
- 外部 API 集成
- MCP Tool 实现（import_questions）

**阶段 7 - 统计分析服务**：
- 统计服务（src/services/statistics_service.py）
- MCP Tools 实现（5 个统计分析 Tools）

**阶段 8 - 集成与测试**：
- 业务编排层（src/services/orchestrator.py）
- 统一参数验证
- 统一错误处理
- 端到端测试
- 性能测试
- 文档编写

### 13.4 下一步开发计划

**即将开始**：阶段 2 - 数据库设计与初始化

**优先级顺序**：
1. 创建 .env.example 文件
2. 定义数据模型（models.py）
3. 实现 SQLite DAO（sqlite_dao.py）
4. 实现 ChromaDB DAO（chroma_dao.py）
5. 实现数据库管理器（database_manager.py）
6. 编写数据库初始化脚本
7. 编写单元测试

**预计完成时间**：1 个工作日

### 13.5 技术栈落地情况

| 技术组件 | 预期版本 | 当前状态 | 说明 |
|---------|---------|---------|------|
| Python | 3.8+ | ✅ 已配置 | 项目运行环境 |
| FastMCP | >=0.2.0 | ✅ 已集成 | MCP Server 框架，基础 Tools 已实现 |
| ChromaDB | >=0.4.0 | ⏳ 已安装，待使用 | 向量数据库，待阶段 2 集成 |
| SQLite | 内置 | ⏳ 待使用 | 元数据库，待阶段 2 集成 |
| Requests | >=2.31.0 | ✅ 已安装 | HTTP 客户端，待阶段 3/6 使用 |
| Pydantic | >=2.5.0 | ✅ 已安装 | 数据验证，待阶段 2 使用 |
| PyYAML | >=6.0.1 | ✅ 已使用 | 配置文件解析 |
| python-dotenv | >=1.0.0 | ✅ 已使用 | 环境变量加载 |

### 13.6 已实现功能验证

**当前可用功能**：

1. ✅ 系统启动和关闭
2. ✅ 配置加载和验证
3. ✅ 日志记录（文件、控制台、敏感信息脱敏）
4. ✅ MCP Server 基础运行
5. ✅ 健康检查 Tool
6. ✅ 服务器信息查询 Tool
7. ✅ 配置测试 Tool

**待验证功能**（依赖后续阶段）：

1. ⏳ 题目创建、查询、更新、删除
2. ⏳ 语义检索和关键词检索
3. ⏳ 批量导入
4. ⏳ 统计分析
5. ⏳ 向量化处理
6. ⏳ 数据一致性保证

### 13.7 质量保证现状

**代码质量**：
- ✅ 完整的类型注解（Type Hints）
- ✅ 详细的文档字符串（Docstrings）
- ✅ 遵循 PEP 8 编码规范
- ✅ 单一职责原则
- ✅ 适当的异常处理

**测试覆盖**：
- ⏸️ 单元测试（待编写）
- ⏸️ 集成测试（待编写）
- ⏸️ 性能测试（待编写）

**文档完善度**：
- ✅ 设计文档（本文档）
- ✅ 代码注释
- ⏸️ API 文档（待编写）
- ⏸️ 用户手册（待编写）
- ⏸️ 部署文档（待编写）

### 13.8 项目里程碑

```mermaid
timeline
    title 项目开发里程碑
    section 已完成
        项目初始化 : 环境搭建
                    : 基础框架
                    : 配置和日志系统
    section 进行中
        数据层开发 : 数据库设计
                    : DAO层实现
    section 待开始
        API集成 : Embedding服务
        核心功能 : 管理服务
                 : 检索服务
        扩展功能 : 批量导入
                 : 统计分析
        交付准备 : 集成测试
                 : 文档完善
```

### 13.9 风险与挑战

**当前阶段面临的挑战**：

1. **数据一致性实现**（中等难度）
   - 需要协调 SQLite 和 ChromaDB 两个数据库
   - 实现可靠的两阶段提交机制
   - 处理各种失败场景

2. **Embedding API 选择**（低难度）
   - 需要评估不同服务商的性价比
   - 考虑中文处理能力
   - API 调用成本控制

3. **性能优化**（中等难度）
   - 向量检索性能优化
   - SQL 查询优化
   - 缓存策略设计

**应对措施**：
- 按阶段逐步实现，降低复杂度
- 充分测试每个阶段的交付物
- 保持代码质量和文档同步
- 及时记录技术决策和经验教训